package ai.accurat.sdk.managers;

import android.text.TextUtils;

import java.util.ArrayList;
import java.util.Map;

import ai.accurat.sdk.core.AccuratLogger;
import ai.accurat.sdk.data.RealmHelper;
import ai.accurat.sdk.data.models.AccuratConfiguration;
import ai.accurat.sdk.data.models.AccuratFeature;
import ai.accurat.sdk.data.models.AccuratFeatureSwitch;
import ai.accurat.sdk.data.models.Setting;

public class AccuratConfigurationManager {

    private static final String TAG = AccuratConfigurationManager.class.getSimpleName();

    public static boolean isValid(AccuratConfiguration configuration) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".isValid()");
        if (configuration == null) {
            AccuratLogger.log(AccuratLogger.ERROR, "AccuratConfiguration can't be null");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".isValid()");

            return false;
        }


        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Does storage contain a username? " + (configuration.hasUsername() ? "Yes" : "No"));
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Is the username valid? " + (configuration.hasValidUsername() ? "Yes" : "No"));
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Does storage contain a password? " + (configuration.hasPassword() ? "Yes" : "No"));
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Is the password valid? " + (configuration.hasValidPassword() ? "Yes" : "No"));
        if (!configuration.hasValidUsername() || !configuration.hasValidPassword()) {
            AccuratLogger.log(AccuratLogger.ERROR, "AccuratConfiguration must contain valid credentials");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".isValid()");

            return false;
        }

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "AccuratConfiguration is valid");
        AccuratLogger.log(AccuratLogger.NONE, "-- username = " + configuration.getUsername());
        //noinspection ReplaceAllDot
        AccuratLogger.log(AccuratLogger.NONE, "-- password = " + configuration.getPassword().replaceAll(".", "*"));
        AccuratLogger.log(AccuratLogger.NONE, "-- notificationTarget = " + configuration.getNotificationTargetClass());
        AccuratLogger.log(AccuratLogger.NONE, "-- features = [" + TextUtils.join(", ", configuration.getFeatures()) + "]");
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".isValid()");

        return true;
    }

    public static void store(AccuratConfiguration configuration) {
        if (configuration == null) {
            return;
        }

        RealmHelper.storeSetting(Setting.Keys.Config.USERNAME, configuration.getUsername());
        RealmHelper.storeSetting(Setting.Keys.Config.PASSWORD, configuration.getPassword());
        RealmHelper.storeSetting(Setting.Keys.Config.NOTIFICATION_TARGET_PACKAGE, configuration.getNotificationTargetPackage());
        RealmHelper.storeSetting(Setting.Keys.Config.NOTIFICATION_TARGET_CLASS, configuration.getNotificationTargetClass());
        RealmHelper.storeSetting(Setting.Keys.Config.GDPR_CONSENT, configuration.usesGdprConsentFeature());
        RealmHelper.storeSetting(Setting.Keys.Config.LOCATION_PERMISSION, configuration.usesLocationPermissionFeature());
        RealmHelper.storeSetting(Setting.Keys.Config.APP_VERSION, configuration.getAppVersion());
        RealmHelper.storeSetting(Setting.Keys.Config.HAS_DEBUG_LOGS_ENABLED, configuration.hasDebugLogsEnabled());
        RealmHelper.storeSetting(Setting.Keys.Config.APP_VERSION_CODE, configuration.getAppVersionCode());
        ArrayList<String> featureSwitches = new ArrayList<>();
        for (Map.Entry<String, Boolean> entry : configuration.getFeatureSwitches().entrySet()) {
            if (entry.getValue()) {
                featureSwitches.add(entry.getKey());
            }
        }
        RealmHelper.storeSetting(Setting.Keys.Config.FEATURE_SWITCHES, String.join(",", featureSwitches));
    }

    public static AccuratConfiguration load() {
        String[] rawFeatureSwitches = RealmHelper
                .loadStringSetting(Setting.Keys.Config.FEATURE_SWITCHES, "")
                .split(",");
        ArrayList<AccuratFeatureSwitch> featureSwitches = new ArrayList<>();
        for (String rawFeatureSwitch : rawFeatureSwitches) {
            try {
                AccuratFeatureSwitch featureSwitch = AccuratFeatureSwitch.valueOf(rawFeatureSwitch);
                featureSwitches.add(featureSwitch);
            } catch (Exception e) {
                // Ignore
            }
        }

        return new AccuratConfiguration()
                .withCredentials(RealmHelper.loadStringSetting(Setting.Keys.Config.USERNAME), RealmHelper.loadStringSetting(Setting.Keys.Config.PASSWORD))
                .withNotificationTargetPackage(RealmHelper.loadStringSetting(Setting.Keys.Config.NOTIFICATION_TARGET_PACKAGE))
                .withNotificationTargetClass(RealmHelper.loadStringSetting(Setting.Keys.Config.NOTIFICATION_TARGET_CLASS))
                .withFeatures(
                        RealmHelper.loadBooleanSetting(Setting.Keys.Config.GDPR_CONSENT) ? AccuratFeature.GDPR_CONSENT : null,
                        RealmHelper.loadBooleanSetting(Setting.Keys.Config.LOCATION_PERMISSION) ? AccuratFeature.LOCATION_PERMISSION : null
                )
                .withAppVersion(RealmHelper.loadStringSetting(Setting.Keys.Config.APP_VERSION))
                .withDebugLogs(RealmHelper.loadBooleanSetting(Setting.Keys.Config.HAS_DEBUG_LOGS_ENABLED))
                .withAppVersionCode(RealmHelper.loadIntSetting(Setting.Keys.Config.APP_VERSION_CODE))
                .withFeatureSwitches(featureSwitches.toArray(new AccuratFeatureSwitch[0]));
    }

    public static String getUsername() {
        return RealmHelper.loadStringSetting(Setting.Keys.Config.USERNAME, "");
    }

    public static String getPassword() {
        return RealmHelper.loadStringSetting(Setting.Keys.Config.PASSWORD, "");
    }

    public static String getNotificationTargetPackage() {
        return RealmHelper.loadStringSetting(Setting.Keys.Config.NOTIFICATION_TARGET_PACKAGE, "");
    }

    public static String getNotificationTargetClass() {
        return RealmHelper.loadStringSetting(Setting.Keys.Config.NOTIFICATION_TARGET_CLASS, "");
    }

    public static boolean hasGdprConsentFeature() {
        return RealmHelper.loadBooleanSetting(Setting.Keys.Config.GDPR_CONSENT, false);
    }

    public static boolean hasLocationPermissionFeature() {
        return RealmHelper.loadBooleanSetting(Setting.Keys.Config.LOCATION_PERMISSION, false);
    }

    public static String loadConsentJson() {
        return RealmHelper.loadStringSetting(Setting.Keys.State.CONSENT_JSON, "");
    }

    public static void storeConsentJson(String consentJson) {
        RealmHelper.storeSetting(Setting.Keys.State.CONSENT_JSON, consentJson);
    }

    public static String getAppVersion() {
        return RealmHelper.loadStringSetting(Setting.Keys.Config.APP_VERSION);
    }

    public static boolean hasDebugLogsEnabled() {
        return RealmHelper.loadBooleanSetting(Setting.Keys.Config.HAS_DEBUG_LOGS_ENABLED, false);
    }

    public static int getAppVersionCode() {
        return RealmHelper.loadIntSetting(Setting.Keys.Config.APP_VERSION_CODE);
    }
}
