package ai.accurat.sdk.core;

import android.location.Location;
import android.text.TextUtils;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.Date;

import ai.accurat.sdk.constants.AccuratLocationKeys;

/**
 * Location object with metadata.
 * Based on OpenLocateLocation.
 *
 * @since 2.0.0
 */
public class AccuratLocation implements LocationInterface {

    private static final String TAG = AccuratLocation.class.getSimpleName();
    private static final String ADVERTISING_ID_TYPE = "aaid";

    private Date created;
    private LocationInfo locationInfo;
    private AdvertisingIdClient.Info advertisingInfo;
    private InformationFields informationFields;
    private UserActivity userActivity;

    @Override
    public Date getCreated() {
        return created;
    }

    @Override
    public void setCreated(Date created) {
        this.created = created;
    }

    @Override
    public LocationInfo getLocationInfo() {
        return locationInfo;
    }

    @Override
    public void setLocationInfo(LocationInfo locationInfo) {
        this.locationInfo = locationInfo;
    }

    @Override
    public AdvertisingIdClient.Info getAdvertisingInfo() {
        return advertisingInfo;
    }

    @Override
    public void setAdvertisingInfo(AdvertisingIdClient.Info advertisingInfo) {
        this.advertisingInfo = advertisingInfo;
    }

    @Override
    public InformationFields getInformationFields() {
        return informationFields;
    }

    @Override
    public void setInformationFields(InformationFields informationFields) {
        this.informationFields = informationFields;
    }

    @Override
    public UserActivity getUserActivity() {
        return userActivity;
    }

    @Override
    public void setUserActivity(UserActivity userActivity) {
        this.userActivity = userActivity;
    }

    AccuratLocation(Location location, AdvertisingIdClient.Info advertisingInfo, InformationFields informationFields) {
        this(location, advertisingInfo, informationFields, informationFields != null ? informationFields.getUserActivity() : null);
    }

    AccuratLocation(Location location, AdvertisingIdClient.Info advertisingInfo, InformationFields informationFields, UserActivity userActivity) {
        this.created = new Date();
        if (location != null) {
            this.locationInfo = new LocationInfo(location);
        }
        this.advertisingInfo = advertisingInfo;
        this.informationFields = informationFields;
        this.userActivity = userActivity;

    }

    AccuratLocation(Date created, String jsonString) {
        this.created = created;
        try {
            JSONObject json = new JSONObject(jsonString);

            locationInfo = new LocationInfo();
            locationInfo.setLatitude(json.getDouble(AccuratLocationKeys.LATITUDE));
            locationInfo.setLongitude(json.getDouble(AccuratLocationKeys.LONGITUDE));
            locationInfo.setHorizontalAccuracy(Float.parseFloat(json.getString(AccuratLocationKeys.HORIZONTAL_ACCURACY)));
            locationInfo.setTimeStampSecs(json.getLong(AccuratLocationKeys.TIMESTAMP));
            locationInfo.setAltitude(json.getDouble(AccuratLocationKeys.ALTITUDE));
            locationInfo.setCourse(Float.parseFloat(json.getString(AccuratLocationKeys.COURSE)));
            locationInfo.setSpeed(Float.parseFloat(json.getString(AccuratLocationKeys.SPEED)));

            String deviceManufacturer = "";
            if (json.has(AccuratLocationKeys.DEVICE_MANUFACTURER)) {
                deviceManufacturer = json.getString(AccuratLocationKeys.DEVICE_MANUFACTURER);
            }


            String deviceModel = "";
            if (json.has(AccuratLocationKeys.DEVICE_MODEL)) {
                deviceModel = json.getString(AccuratLocationKeys.DEVICE_MODEL);
            }

            String chargingState = "";
            if (json.has(AccuratLocationKeys.IS_CHARGING)) {
                chargingState = json.getString(AccuratLocationKeys.IS_CHARGING);
            }

            String operatingSystem = "";
            if (json.has(AccuratLocationKeys.OPERATING_SYSTEM)) {
                operatingSystem = json.getString(AccuratLocationKeys.OPERATING_SYSTEM);
            }

            String carrierName = "";
            if (json.has(AccuratLocationKeys.CARRIER_NAME)) {
                carrierName = json.getString(AccuratLocationKeys.CARRIER_NAME);
            }

            String wifiSSID = "";
            if (json.has(AccuratLocationKeys.WIFI_SSID)) {
                wifiSSID = json.getString(AccuratLocationKeys.WIFI_SSID);
            }

            String wifiBSSID = "";
            if (json.has(AccuratLocationKeys.WIFI_BSSID)) {
                wifiBSSID = json.getString(AccuratLocationKeys.WIFI_BSSID);
            }

            String connectionType = "";
            if (json.has(AccuratLocationKeys.CONNECTION_TYPE)) {
                connectionType = json.getString(AccuratLocationKeys.CONNECTION_TYPE);
            }

            String locationMethod = "";
            if (json.has(AccuratLocationKeys.LOCATION_METHOD)) {
                locationMethod = json.getString(AccuratLocationKeys.LOCATION_METHOD);
            }

            String locationContext = "";
            if (json.has(AccuratLocationKeys.LOCATION_CONTEXT)) {
                locationContext = json.getString(AccuratLocationKeys.LOCATION_CONTEXT);
            }

            informationFields = InformationFieldsFactory.getInformationFields(deviceManufacturer, deviceModel, chargingState, operatingSystem, carrierName, wifiSSID, wifiBSSID, connectionType, locationMethod, locationContext, userActivity);

            if (json.has(AccuratLocationKeys.AD_ID) && json.has(AccuratLocationKeys.AD_OPT_OUT)) {
                advertisingInfo = new AdvertisingIdClient.Info(
                        json.getString(AccuratLocationKeys.AD_ID),
                        json.getBoolean(AccuratLocationKeys.AD_OPT_OUT)
                );
            } else {
                advertisingInfo = null;
            }

        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + "(): " + e.getMessage());
        }
    }

    @Override
    public JSONObject getJson() {
        JSONObject jsonObject = new JSONObject();

        try {
            if (locationInfo != null) {
                jsonObject
                        .put(AccuratLocationKeys.LATITUDE, locationInfo.getLatitude())
                        .put(AccuratLocationKeys.LONGITUDE, locationInfo.getLongitude())
                        .put(AccuratLocationKeys.HORIZONTAL_ACCURACY, String.valueOf(locationInfo.getHorizontalAccuracy()))
                        .put(AccuratLocationKeys.TIMESTAMP, locationInfo.getTimeStampSecs())
                        .put(AccuratLocationKeys.COURSE, String.valueOf(locationInfo.getCourse()))
                        .put(AccuratLocationKeys.SPEED, String.valueOf(locationInfo.getSpeed()))
                        .put(AccuratLocationKeys.ALTITUDE, locationInfo.getAltitude());
            }

            // Add the advertising info, if available
            if (advertisingInfo != null) {
                jsonObject
                        .put(AccuratLocationKeys.AD_ID, advertisingInfo.getId())
                        .put(AccuratLocationKeys.AD_OPT_OUT, advertisingInfo.isLimitAdTrackingEnabled())
                        .put(AccuratLocationKeys.AD_TYPE, ADVERTISING_ID_TYPE);
            }

            // Add the information fields, if available
            if (informationFields != null) {
                if (!TextUtils.isEmpty(informationFields.getManufacturer()))
                    jsonObject.put(AccuratLocationKeys.DEVICE_MANUFACTURER, informationFields.getManufacturer());


                if (!TextUtils.isEmpty(informationFields.getModel())) {
                    jsonObject.put(AccuratLocationKeys.DEVICE_MODEL, informationFields.getModel());
                }

                if (!TextUtils.isEmpty(informationFields.isCharging())) {
                    jsonObject.put(AccuratLocationKeys.IS_CHARGING, informationFields.isCharging());
                }

                if (!TextUtils.isEmpty(informationFields.getOperatingSystem())) {
                    jsonObject.put(AccuratLocationKeys.OPERATING_SYSTEM, informationFields.getOperatingSystem());
                }

                if (!TextUtils.isEmpty(informationFields.getCarrierName())) {
                    jsonObject.put(AccuratLocationKeys.CARRIER_NAME, informationFields.getCarrierName());
                }

                if (!(TextUtils.isEmpty(informationFields.getWifiSsid()) && TextUtils.isEmpty(informationFields.getWifiBssid()))) {
                    jsonObject.put(AccuratLocationKeys.WIFI_SSID, informationFields.getWifiSsid());
                    jsonObject.put(AccuratLocationKeys.WIFI_BSSID, informationFields.getWifiBssid());
                }

                if (!TextUtils.isEmpty(informationFields.getConnectionType())) {
                    jsonObject.put(AccuratLocationKeys.CONNECTION_TYPE, informationFields.getConnectionType());
                }

                if (informationFields.getLocationProvider() != null) {
                    if (!TextUtils.isEmpty(informationFields.getLocationProvider().getValue())) {
                        jsonObject.put(AccuratLocationKeys.LOCATION_METHOD, informationFields.getLocationProvider().getValue());
                    }

                    if (!TextUtils.isEmpty(informationFields.getLocationContext().getValue())) {
                        jsonObject.put(AccuratLocationKeys.LOCATION_CONTEXT, informationFields.getLocationContext().getValue());
                    }
                }
            }

            // Add the user activity, if available
            if (userActivity != null) {
                jsonObject.put(AccuratLocationKeys.USER_ACTIVITY_TYPE, userActivity.getType());
                jsonObject.put(AccuratLocationKeys.USER_ACTIVITY_CONFIDENCE, userActivity.getConfidence());
            }
        } catch (NullPointerException | JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".getJson(): " + e.getMessage());
        }

        return jsonObject;
    }

    @Override
    public String toString() {
        return "AccuratLocation{" +
                "locationInfo=" + locationInfo +
                ", advertisingInfo=" + advertisingInfo +
                ", informationFields=" + informationFields +
                '}';
    }
}
