package ai.accurat.sdk.core;

import android.content.Context;
import android.text.TextUtils;

import androidx.annotation.NonNull;

import com.android.volley.Request;
import com.android.volley.RequestQueue;
import com.android.volley.toolbox.JsonObjectRequest;
import com.android.volley.toolbox.Volley;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;

import ai.accurat.sdk.config.Configuration;
import ai.accurat.sdk.constants.AccuratEndpoints;
import ai.accurat.sdk.constants.ApiKeys;
import ai.accurat.sdk.constants.HttpMethod;
import ai.accurat.sdk.constants.StorageKeys;

/**
 * @author Kenneth
 * @since 2020-06-23
 */
public class GdprRightsManager {

    // <editor-fold desc="Fields">
    private static final String TAG = GdprRightsManager.class.getSimpleName();

    private static MultiProcessStorage storage;
    private static RequestQueue requestQueue;
    // </editor-fold>

    // <editor-fold desc="Initialisation">
    public static void init(@NonNull Context context) {
        if (!isInitialized()) {
            AccuratLogger.init(context);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Initialising " + TAG);
            storage = MultiProcessStorage.getStorage(context, StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE);
            requestQueue = Volley.newRequestQueue(context);
        } else if (requestQueue == null) {
            requestQueue = Volley.newRequestQueue(context);
        }
    }

    private static boolean isInitialized() {
        return storage != null;
    }

    private static void checkInitialized() {
        if (!isInitialized()) {
            throw new IllegalStateException(TAG + " has not yet been initialised.");
        }
    }
    // </editor-fold>

    public static void invokeRight(String right, String info) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".invokeRight()");
        if (TextUtils.isEmpty(right)) {
            AccuratLogger.log(AccuratLogger.WARNING, "right should not be null or empty");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".invokeRight()");

            return;
        }

        String adId = AdvertisingManager.getAdId();
        if (TextUtils.isEmpty(adId)) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "No ad ID available yet, can't call invokeRight()");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".invokeRight()");

            return;
        }

        HashMap<String, Object> urlParameters = new HashMap<>();
        urlParameters.put(ApiKeys.Url.AD_ID, adId);

        JSONObject requestBody = getRequestBody(right, info);
        if (requestBody == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "Couldn't create request body, can't call invokeRight()");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".invokeRight()");

            return;
        }

        JsonObjectRequest invokeRightRequest = new JsonObjectRequest(
                Request.Method.POST,
                AccuratEndpoints.POST_GDPR_RIGHT.getUrl(urlParameters),
                requestBody,
                response -> AccuratLogger.logNetworkResponse(HttpMethod.POST, Configuration.ENDPOINT_POST_GDPR_RIGHT, response, false),
                error -> AccuratLogger.logNetworkError(HttpMethod.POST, Configuration.ENDPOINT_POST_GDPR_RIGHT, error)
        ) {
            @Override
            public Map<String, String> getHeaders() {
                return AccuratApi.getHeaders(
                        storage,
                        "POST",
                        getBodyContentType(),
                        AccuratApi.getEncodedRequestBody(requestBody.toString()),
                        AccuratEndpoints.POST_GDPR_RIGHT.getPath(urlParameters)
                );
            }
        };

        invokeRightRequest.setTag(TAG)
                .setRetryPolicy(AccuratApi.defaultRetryPolicy)
                .setShouldCache(false);

        if(requestQueue == null) {
            AccuratLogger.log(AccuratLogger.ERROR, "Failed to make API call, requestQueue is NULL");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".invokeRight()");

            return;
        }

        AccuratLogger.logNetworkRequest(HttpMethod.POST, Configuration.ENDPOINT_POST_GDPR_RIGHT, requestBody, false);
        requestQueue.add(invokeRightRequest);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".invokeRight()");
    }

    private static JSONObject getRequestBody(String right, String info) {
        JSONObject requestParameters = new JSONObject();
        try {
            requestParameters.put("right", right);
            requestParameters.put("info", info);

            return requestParameters;
        } catch (JSONException e) {
            return null;
        }
    }
}
