package ai.accurat.sdk.core;

import android.text.TextUtils;
import android.util.LongSparseArray;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.Locale;

import ai.accurat.sdk.constants.StorageKeys;

public class GeofenceNotificationLog extends JSON {

    // <editor-fold desc="Constants">
    private static final String TAG = GeofenceNotificationLog.class.getSimpleName();
    public static final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd", Locale.getDefault());
    public static final SimpleDateFormat TIME_FORMAT = new SimpleDateFormat("HH:mm:ss", Locale.getDefault());
    // </editor-fold>

    // <editor-fold desc="Fields">
    private LongSparseArray<LongSparseArray<Integer>> dayNotificationCounts;
    // </editor-fold>

    // <editor-fold desc="Initialisation">
    public GeofenceNotificationLog() {
        dayNotificationCounts = new LongSparseArray<>();
    }

    public GeofenceNotificationLog(LongSparseArray<LongSparseArray<Integer>> dayNotificationCounts, LongSparseArray<Integer> dayCounts) {
        this.dayNotificationCounts = dayNotificationCounts;
    }

    public GeofenceNotificationLog(JSONObject json) {
        update(json);
    }
    // </editor-fold>

    // <editor-fold desc="JSON Handling">

    /**
     * JSON structure of a {@link GeofenceNotificationLog}-object:
     * <pre>
     * {
     *   "date_yyyy-MM-DD_1": [
     *     {
     *       "nid": long,
     *       "count": int
     *     },
     *     ...
     *   ],
     *   "date_yyyy-MM-DD_2": [...],
     *   ...
     * }
     * </pre>
     *
     * @param json
     */
    @Override
    public void update(JSONObject json) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".update()");
        dayNotificationCounts = new LongSparseArray<>();

        // Iterate over all dates
        Iterator<String> keyIterator = json.keys();
        while (keyIterator.hasNext()) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- Starting an iteration");
            // Get the date
            String dateKey = keyIterator.next();
            AccuratLogger.log(AccuratLogger.NONE, "dateKey = " + dateKey);
            long date = dateToLong(dateKey);
            AccuratLogger.log(AccuratLogger.NONE, "date = " + date);
            if (date < 0) {
                // Skip to next date, unparsable dateKey
                AccuratLogger.log(AccuratLogger.NONE, "=> Unparsable date key, skipping to next date");
                continue;
            }

            JSONArray notificationCountsArray = json.optJSONArray(dateKey);
            if (notificationCountsArray == null || notificationCountsArray.length() == 0) {
                // Skip to the next date, if no mappings are available.
                // This assumes dayCount is also 0.
                AccuratLogger.log(AccuratLogger.NONE, "=> No mappings available, skipping to next date");
                continue;
            }

            LongSparseArray<Integer> notificationCounts = new LongSparseArray<>();
            AccuratLogger.log(AccuratLogger.NONE, "Going to iterate all notificationId => count mappings");
            // Iterate over all notificationId => count mappings
            for (int i = 0; i < notificationCountsArray.length(); i++) {
                AccuratLogger.log(AccuratLogger.NONE, "... i = " + 0);
                JSONObject object = notificationCountsArray.optJSONObject(i);
                if (object == null) {
                    // Skip to the next mapping if this one is invalid
                    AccuratLogger.log(AccuratLogger.NONE, "... mapping is invalid, skipping to next one");
                    continue;
                }

                long notificationId = object.optLong(StorageKeys.GeofenceNotificationLog.NOTIFICATION_ID);
                if (notificationId == 0L) {
                    // Skip to the next mapping if the notificationId is unusable
                    AccuratLogger.log(AccuratLogger.NONE, "... notificationId is unusable, skipping to next one");
                    continue;
                }
                int count = object.optInt(StorageKeys.GeofenceNotificationLog.COUNT);
                if (count <= 0) {
                    // Skip to the next mapping if the count is unnecessary
                    AccuratLogger.log(AccuratLogger.NONE, "... count is <= 0, skipping to next one");
                    continue;
                }
                // Add the mapping
                AccuratLogger.log(AccuratLogger.NONE, "Found a mapping (" + notificationId + ", " + count + "), using it");
                notificationCounts.put(notificationId, count);
                AccuratLogger.log(AccuratLogger.NONE, "=> Used, continuing with next one");
            }
            AccuratLogger.log(AccuratLogger.NONE, "Done looping, found " + notificationCounts.size() + " notifications");

            if (notificationCounts.size() == 0) {
                // Skip to the next date if no valid mappings are available
                AccuratLogger.log(AccuratLogger.NONE, "=> No valid mappings, skipping to next date");
                continue;
            }

            // Add the mappings to the date
            AccuratLogger.log(AccuratLogger.NONE, "Adding all mappings to the date");
            dayNotificationCounts.put(date, notificationCounts);

            AccuratLogger.log(AccuratLogger.NONE, "=> Done with this date, continuing with next date");
        }
        AccuratLogger.log(AccuratLogger.NONE, "Done with dates");
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".update()");
    }

    /**
     * JSON structure of a {@link GeofenceNotificationLog}-object:
     * <pre>
     * {
     *   "date_yyyy-MM-DD_1": [
     *     {
     *       "nid": long,
     *       "count": int
     *     },
     *     ...
     *   ],
     *   "date_yyyy-MM-DD_2": [...],
     *   ...
     * }
     * </pre>
     *
     * @return
     */
    @Override
    public JSONObject toJson() {
        JSONObject json = new JSONObject();

        // Iterate over all dates in the map
        for (int i = 0; i < dayNotificationCounts.size(); i++) {
            long date = dayNotificationCounts.keyAt(i);
            // Get the notificationId => count mappings for date
            LongSparseArray<Integer> counts = dayNotificationCounts.get(date);
            if (counts == null || counts.size() == 0) {
                // Skip to the next date, if no mappings were found
                continue;
            }

            JSONArray countsArray = new JSONArray();
            // Iterate over all notificationIds
            for (int j = 0; j < counts.size(); j++) {
                long notificationId = counts.keyAt(j);
                int count = counts.get(notificationId);
                if (count <= 0) {
                    // Skip to the next notificationId if count is unnecessary/unusable
                    continue;
                }

                JSONObject object = new JSONObject();
                try {
                    object.put(StorageKeys.GeofenceNotificationLog.NOTIFICATION_ID, notificationId);
                    object.put(StorageKeys.GeofenceNotificationLog.COUNT, count);
                } catch (JSONException e) {
                    AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".toJson(): " + e.getMessage());
                    e.printStackTrace();
                }
                if (object.length() != 2) {
                    // Skip to the next notificationId if the object can't be build.
                    // This should not happen, but we're programming defensively.
                    continue;
                }
                countsArray.put(object);
            }

            if (countsArray.length() == 0) {
                // Skip to the next date if no mappings were added
                continue;
            }

            try {
                // A the date with mappings to the json output
                json.put(longToDate(date), countsArray);
            } catch (JSONException e) {
                AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".toJson(): " + e.getMessage());
                e.printStackTrace();
            }
        }

        return json;
    }
    // </editor-fold>

    // <editor-fold desc="Getters">
    public int getNotificationCountForDate(Date date) {
        return getNotificationCountForDate(DATE_FORMAT.format(date));
    }

    public int getNotificationCountForDate(String date) {
        long dateLong = dateToLong(date);
        if (dateLong < 0) {
            return 0;
        }

        LongSparseArray<Integer> notificationCounts = dayNotificationCounts.get(dateLong);
        if(notificationCounts == null || notificationCounts.size() == 0) {
            return 0;
        }

        int count = 0;
        for (int i = 0; i < notificationCounts.size(); i++) {
            count += notificationCounts.get(notificationCounts.keyAt(i));
        }

        return count;
    }

    public String getMostRecentDateForNotificationId(long notificationId) {
        String sinceToday = DATE_FORMAT.format(new Date());

        return getMostRecentDateForNotificationId(notificationId, sinceToday);
    }

    public String getMostRecentDateForNotificationId(long notificationId, Date sinceDate) {
        return getMostRecentDateForNotificationId(notificationId, DATE_FORMAT.format(sinceDate));
    }

    public String getMostRecentDateForNotificationId(long notificationId, String sinceDate) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".getMostRecentDateForNotificationId()");
        long date = dateToLong(sinceDate);
        AccuratLogger.log(AccuratLogger.NONE, "sinceDate = " + sinceDate);
        if (date < 0) {
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getMostRecentDateForNotificationId()");
            return null;
        }

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Iterating over dates");
        // Iterate over all dates from most recent to oldest
        for (int i = dayNotificationCounts.size() - 1; i >= 0; i--) {
            long loopDate = dayNotificationCounts.keyAt(i);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "loopDate = " + loopDate);
            if (loopDate > date) {
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "loopDate > date => Continuing");
                continue;
            }

            LongSparseArray<Integer> notificationCounts = dayNotificationCounts.get(loopDate);
            if(notificationCounts == null) {
                AccuratLogger.log(AccuratLogger.NONE, "notificationCounts is null");
                return null;
            }
            AccuratLogger.log(AccuratLogger.NONE, "notificationCounts = " + notificationCounts.toString());

            if (notificationCounts.indexOfKey(notificationId) >= 0 && notificationCounts.get(notificationId) > 0) {
                AccuratLogger.log(AccuratLogger.NONE, "Returning " + longToDate(loopDate));
                AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getMostRecentDateForNotificationId()");
                return longToDate(loopDate);
            }
        }

        AccuratLogger.log(AccuratLogger.NONE, "Returning null");
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".getMostRecentDateForNotificationId()");
        return null;
    }
    // </editor-fold>

    // <editor-fold desc="Setters">
    public boolean addNotification(long notificationId, Date date) {
        return addNotification(
                notificationId,
                DATE_FORMAT.format(date),
                getMostRecentDateForNotificationId(notificationId, date)
        );
    }

    public boolean addNotification(long notificationId, String dateString, String previousDateString) {
        long date = dateToLong(dateString);
        if (date < 0) {
            // Invalid date
            return false;
        }

        long previousDate = dateToLong(previousDateString);
        if (date < previousDate) {
            // Trying to add a notification on an older date than the most recent one.
            return false;
        }

        // Update the count
        LongSparseArray<Integer> notificationCounts = dayNotificationCounts.get(date);
        if (notificationCounts == null) {
            notificationCounts = new LongSparseArray<>();
            notificationCounts.put(notificationId, 1);
        } else if (notificationCounts.indexOfKey(notificationId) < 0) {
            notificationCounts.put(notificationId, 1);
        } else {
            notificationCounts.put(notificationId, notificationCounts.get(notificationId) + 1);
        }
        dayNotificationCounts.put(date, notificationCounts);

        if (date > previousDate) {
            clearNotification(notificationId, previousDate);
        }

        return true;
    }

    private void clearNotification(long notificationId, long date) {
        LongSparseArray<Integer> notificationCounts = dayNotificationCounts.get(date);
        if (notificationCounts == null || notificationCounts.size() == 0) {
            return;
        }

        notificationCounts.delete(notificationId);
    }
    // </editor-fold>

    // <editor-fold desc="Helpers">
    private static long dateToLong(String date) {
        if (!isValidDate(date)) {
            return -1;
        }

        String[] dateParts = date.split("-");
        long year = Long.parseLong(dateParts[0]);
        int month = Integer.parseInt(dateParts[1]);
        int day = Integer.parseInt(dateParts[2]);

        return year * 10000 + month * 100 + day;// String yyyy-MM-DD => long yyyyMMDD
    }

    private static String longToDate(long date) {
        if (date < 0) {
            return null;
        }

        long day = date % 100;
        long month = (date / 100) % 100;
        long year = (date / 10000);

        return year + "-" + (month < 10 ? "0" : "") + month + "-" + (day < 10 ? "0" : "") + day;// yyyy-MM-DD
    }

    /**
     * Checks if a date String uses the format yyyy-MM-DD.
     * This method doesn't check for a valid date! It only verifies that this date
     * is usable by the {@link GeofenceNotificationLog#dateToLong}
     *
     * @param date The date {@link String}
     * @return boolean
     */
    public static boolean isValidDate(String date) {
        if (TextUtils.isEmpty(date)) {
            return false;
        }

        String[] dateParts = date.split("-");
        if (dateParts.length != 3) {
            return false;
        }

        if (dateParts[0].length() < 4) {
            return false;
        }

        if (dateParts[1].length() != 2) {
            return false;
        }

        if (dateParts[2].length() != 2) {
            return false;
        }

        try {
            long year = Long.parseLong(dateParts[0]);
            if (year < 0) {
                return false;
            }
        } catch (NumberFormatException e) {
            return false;
        }

        try {
            int month = Integer.parseInt(dateParts[1]);
            if (month < 1 || month > 12) {
                return false;
            }
        } catch (NumberFormatException e) {
            return false;
        }

        try {
            int day = Integer.parseInt(dateParts[2]);
            if (day < 1 || day > 31) {
                return false;
            }
        } catch (NumberFormatException e) {
            return false;
        }

        return true;
    }
    // </editor-fold>
}
