package ai.accurat.sdk.core;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.graphics.Color;
import android.location.Location;
import android.os.Build;
import android.text.TextUtils;
import android.util.LongSparseArray;

import androidx.annotation.NonNull;
import androidx.core.app.ActivityCompat;
import androidx.core.app.NotificationCompat;
import androidx.core.app.NotificationManagerCompat;

import com.android.volley.AuthFailureError;
import com.android.volley.DefaultRetryPolicy;
import com.android.volley.Request;
import com.android.volley.RequestQueue;
import com.android.volley.toolbox.JsonArrayRequest;
import com.android.volley.toolbox.JsonObjectRequest;
import com.android.volley.toolbox.Volley;
import com.google.android.gms.common.util.CollectionUtils;
import com.google.android.gms.location.Geofence;
import com.google.android.gms.location.GeofencingClient;
import com.google.android.gms.location.GeofencingRequest;
import com.google.android.gms.location.LocationServices;
import com.google.android.gms.maps.model.LatLng;
import com.google.gson.Gson;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import ai.accurat.sdk.R;
import ai.accurat.sdk.callbacks.AccuratActionableCallback;
import ai.accurat.sdk.callbacks.AccuratCompletionCallback;
import ai.accurat.sdk.callbacks.AccuratProcessCallback;
import ai.accurat.sdk.config.Configuration;
import ai.accurat.sdk.constants.AccuratEndpoints;
import ai.accurat.sdk.constants.AccuratLanguage;
import ai.accurat.sdk.constants.ApiKeys;
import ai.accurat.sdk.constants.HttpMethod;
import ai.accurat.sdk.constants.RequestCodes;
import ai.accurat.sdk.constants.ServerDataKeys;
import ai.accurat.sdk.constants.StorageKeys;
import ai.accurat.sdk.managers.AccuratConfigurationManager;
import ai.accurat.sdk.managers.RealmManager;
import io.reactivex.BackpressureStrategy;
import io.reactivex.disposables.CompositeDisposable;
import io.reactivex.schedulers.Schedulers;
import io.reactivex.subjects.PublishSubject;

/**
 * Manager class for fetching, storing and updating geofences.
 *
 * @Accurat
 * @since 2.0.0
 */
public final class GeofencesManager {

    //<editor-fold desc="Fields">
    public static final String EXTRA_NOTIFICATION_ID = "notification_id_geofence_accurat";
    public static final String EXTRA_NOTIFICATION_DATA = "notification_data_geofence_accurat";
    private static final String TAG = GeofencesManager.class.getSimpleName();
    private static final String JSON_ROOT = "data";
    private static final String JSON_GEOFENCE_CHECKSUM = "checksum";
    private static final String JSON_GEOFENCE_STATUS_KEY = "status";
    private static final String JSON_GEOFENCE_STATUS_UPDATED = "changed";
    private static final String POST_BODY_PARAM_LANGUAGE = "lang";
    private static final String POST_BODY_PARAM_NIDS = "nids";
    private static final String POST_BODY_PARAM_ID = "id";

    private static MultiProcessStorage mStorage;
    private static RequestQueue requestQueue;
    private static String notificationChannelId;

    private static GeofencingClient geofencingClient;
    private static final String CURRENT_LOCATION_GEOFENCE_ID_PREFIX = "current_location_fence_accurat_";
    private static final String META_GEOFENCE_ID = "meta_fence_accurat";
    private static AccuratGeofence metaGeofence; // geofence surrounding all the other geofences to
    private static List<AccuratGeofence> currentMonitoredGeofences = new ArrayList<>();
    private static LatLng[] searchBox;

    private static long startTimeGeofenceSearch = 0L;
    private static final int RECENT_GEOFENCES_LIMIT = 100;
    private static final int MAX_NEAR_GEOFENCES = 97;
    private static AccuratGeofenceRange startingSearchBox = new AccuratGeofenceRange(0.0125, 0.02);
    private static final Double MIN_LAT_BOX_RANGE = 0.003; // min range of the bounding box
    public static final Double DISTANCE_CONVERSION_LATITUDE = 0.01;
    public static final Double DISTANCE_CONVERSION_METER = 1112.0;
    public static final Double ERROR_MARGIN = 150.0; // error margin in meters when enter/exit of a geofence is triggered
    public static final Double MIN_META_GEOFENCE_RANGE = 200.0; // min radius of meta geofence in meters
    private static final List<String> CURRENT_LOCATION_FENCE_REQUEST_IDS = new ArrayList<String>() {{
        add(CURRENT_LOCATION_GEOFENCE_ID_PREFIX + 1);
    }};
    private static final Double MAX_META_GEOFENCE_LAT_RANGE = 0.2; // max radius of meta geofence box in lat
    private static final Double MAX_META_GEOFENCE_LNG_RANGE = 0.32; // max radius of meta geofence box in lng
    private static final long GEOFENCE_EXPIRATION_IN_MS = 12 * 60 * 60 * 1000; // 12 hours
    private static final int GEOFENCE_LOITERING_DELAY = 300000; // 5 minutes
    private static final long GEOFENCE_TRIGGER_HIATUS = 3 * 3600; // There must be three hours between notifications of the same geofence

    private static Context lastContext;
    private static PendingIntent geofencePendingIntent;

    private static final PublishSubject<LocationUpdateWrapper> locationUpdateSubject;
    private static final CompositeDisposable compositeDisposable;
    //</editor-fold>

    //<editor-fold desc="Initialisation">
    static {
        locationUpdateSubject = PublishSubject.create();
        compositeDisposable = new CompositeDisposable();
        compositeDisposable.add(locationUpdateSubject.toFlowable(BackpressureStrategy.BUFFER)
                .observeOn(Schedulers.computation())
                .map(GeofencesManager::calculateGeofenceSelection)
                .map(geofenceSelection -> {
                    String logMessage = "It took " + (double) geofenceSelection.getDurationInMs() / 1000 + " seconds and "
                            + geofenceSelection.getNumIterations() + " iterations to find " + geofenceSelection.getGeofences().size()
                            + " geofences with box: [" + geofenceSelection.getBox().getLatitudeRange() + ", " + geofenceSelection.getBox().getLongitudeRange() + "]";
                    AccuratLogger.log(AccuratLogger.GEOFENCE, logMessage);

                    monitorClosestGeofences(lastContext, geofenceSelection);

                    return true;
                })
                .subscribeOn(Schedulers.io())
                .subscribe(hasCompleted -> {
                    AccuratLogger.log(AccuratLogger.SDK_FLOW, "Geofence update flow completed successfully");
                }, throwable -> {
                    AccuratLogger.log(AccuratLogger.ERROR, "Geofence update flow failed: " + throwable.getMessage());
                }));
    }

    public static void init(Context context) {
        if (!isInitialized()) {
            AccuratLogger.init(context);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Initialising " + TAG);
            mStorage = MultiProcessStorage.getStorage(context, StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE);
            requestQueue = Volley.newRequestQueue(context);
            geofencingClient = LocationServices.getGeofencingClient(context);
            createNotificationChannel(context);
        } else if (requestQueue == null) {
            requestQueue = Volley.newRequestQueue(context);
        }
    }

    /**
     * Creates a notification channel on Android O and above (required).
     *
     * @param context Context used to derive channel details and get the NotificationManager.
     */
    private static void createNotificationChannel(Context context) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".createNotificationChannel()");
            final String channelPrefix = "accurat_geofence_id_for_";
            notificationChannelId = channelPrefix + AccuratConfigurationManager.getUsername();
            final NotificationManager notificationManager = context.getSystemService(NotificationManager.class);
            if (notificationManager == null) {
                AccuratLogger.log(AccuratLogger.ERROR, "Could not create notificaiton channel, notificationManager is NULL");
                AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".createNotificationChannel()");

                return;
            }

            // Remove deprecated notification channels
            for (NotificationChannel notificationChannel : notificationManager.getNotificationChannels()) {
                if (!notificationChannel.getId().equals(notificationChannelId)) {
                    notificationManager.deleteNotificationChannel(notificationChannel.getId());
                }
            }

            // Add Geofence notification channel
            CharSequence name = context.getString(R.string.notification_channel_geofences_name);
            String description = context.getString(R.string.notification_channel_geofences_description);
            int importance = NotificationManager.IMPORTANCE_HIGH;
            NotificationChannel channel = new NotificationChannel(notificationChannelId, name, importance);
            channel.setDescription(description);
            // Register the channel with the system
            notificationManager.createNotificationChannel(channel);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Successfully created notificaiton channel");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".createNotificationChannel()");
        }
    }

    private static boolean isInitialized() {
        return mStorage != null;
    }

    private static void checkInitialized() {
        if (!isInitialized()) {
            throw new IllegalStateException("GeofencesManager has not yet been initialised.");
        }
    }
    //</editor-fold>

    //<editor-fold desc="Public interface">

    //<editor-fold desc="Synchronisation">

    /**
     * Updates the geofences in storage.
     *
     * @param context  Used to access the network and storage.
     * @param callback Used to communicate API request and parsing success.
     *                 {@link AccuratActionableCallback#onActionPotentiallyRequired(boolean)} will be used to communicate storage changes.
     */
    public static void sync(final Context context, @NonNull final AccuratActionableCallback callback) {
        sync(context, callback, false);
    }

    /**
     * Updates the geofences in storage.
     *
     * @param context           Used to access the network and storage.
     * @param callback          Used to communicate API request and parsing success.
     *                          {@link AccuratActionableCallback#onActionPotentiallyRequired(boolean)} will be used to communicate storage changes.
     * @param startedFromWorker Should be true when the manager has been started from a periodic build job to avoid endless loops.
     */
    static void sync(final Context context, @NonNull final AccuratActionableCallback callback, boolean startedFromWorker) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".sync()");
        checkInitialized();

        // if geofences are stored locally then verify the status first
        if (hasGeofences(context)) {
            AccuratLogger.log(AccuratLogger.STORAGE, "Geofences found in storage");
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Going to check geofences status");
            fetchStatus(context, callback);
        } else {
            AccuratLogger.log(AccuratLogger.STORAGE, "No geofences in storage");
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Going to fetch geofences");
            // the status call can be omitted if no geofences are present
            fetchGeofences(context, callback);
        }

        // Plan periodic synchronisation of geofences
        if (!startedFromWorker) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Going to plan sync");
            GeofencesSyncWorker.planGeofencesSync();
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".sync()");
    }

    /**
     * Stops ongoing requests to the geofence-related API endpoints.
     */
    public static void stopSync() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".stopSync()");
        if (requestQueue != null) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Stopping requestQueue...");
            requestQueue.stop();
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".stopSync()");
    }
    //</editor-fold>

    //<editor-fold desc="Monitoring">

    /**
     * Starts monitoring of the local geofences.
     *
     * @param context           Used for storage and network access
     * @param lastKnownLocation Used to determine which geofences to track
     * @param callback          Used to communicate whether monitoring has successfully started.
     */
    public static void startTracking(Context context, LocationInterface lastKnownLocation, AccuratProcessCallback callback) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".startTracking()");

        if (lastKnownLocation == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "No last known location yet, won't start tracking geofences");
            callback.onProcessed(false);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".startTracking()");

            return;
        }
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Starting tracking...");

        lastContext = context;
        locationUpdateSubject.onNext(new LocationUpdateWrapper(context, lastKnownLocation));
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".startTracking()");
    }

    public static void stopTracking() {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".stopTracking()");
        clearGeofences(lastContext);
        compositeDisposable.clear();
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".stopTracking()");
    }

    /**
     * Should be called to handle location updates.
     * Will verify whether the list of geofences need to be updated.
     *
     * @param context         Context used to access local storage.
     * @param currentLocation New location.
     */
    public static void onLocationChanged(Context context, LocationInterface currentLocation) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".onLocationChanged()");
        lastContext = context;
        boolean onlyUpdateMetaFences = false;
        // Load data from storage, if needed
        metaGeofence = loadMetaGeofence(metaGeofence);
        currentMonitoredGeofences = loadCurrentMonitoredGeofences(currentMonitoredGeofences);
        if (metaGeofence == null) {
            AccuratLogger.log(AccuratLogger.NONE, "There's no known meta geofence");
            onlyUpdateMetaFences = !CollectionUtils.isEmpty(currentMonitoredGeofences) && currentMonitoredGeofences.size() < MAX_NEAR_GEOFENCES;
            AccuratLogger.log(
                    AccuratLogger.SDK_FLOW,
                    onlyUpdateMetaFences
                            ? "Update current location geofence only"
                            : "Update all geofences"
            );
        } else if (metaGeofence.contains(currentLocation)) {
            AccuratLogger.log(AccuratLogger.NONE, "We're still inside the meta geofence [" + metaGeofence.getId() + "], no need to update geofences");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".onLocationChanged()");

            return;
        } else {
            AccuratLogger.log(AccuratLogger.GEOFENCE, "We've exited the meta geofence [" + metaGeofence.getId() + "], calculating a new set of geofences");
        }

        locationUpdateSubject.onNext(new LocationUpdateWrapper(context, currentLocation, onlyUpdateMetaFences));
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".onLocationChanged()");
    }

    @io.reactivex.annotations.NonNull
    private static GeofenceSelection calculateGeofenceSelection(LocationUpdateWrapper locationUpdateWrapper) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".calculateGeofenceSelection()");

        startTimeGeofenceSearch = System.currentTimeMillis();
        if (locationUpdateWrapper.isUpdateMetaFencesOnly()) {
            GeofenceSelection selection = new GeofenceSelection(locationUpdateWrapper.getLocation(), new ArrayList<>(), startingSearchBox, 0);
            selection.setDurationInMs(System.currentTimeMillis() - startTimeGeofenceSearch);

            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Retuning empty list, only meta geofence needs to be updated");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".calculateGeofenceSelection()");

            return selection;
        }

        GeofenceDataSource dataSource = new AccuratDatabase(DatabaseHelper.getInstance(locationUpdateWrapper.getContext()));
        // Get the 100 most recently added geofences from the database
        List<AccuratGeofence> currentGeofences = dataSource.getRecentGeofences(RECENT_GEOFENCES_LIMIT);
        if (currentGeofences.size() <= MAX_NEAR_GEOFENCES) {
            // If there are at most 97 recent geofences:
            GeofenceSelection selection = new GeofenceSelection(locationUpdateWrapper.getLocation(), currentGeofences, startingSearchBox, 0);
            selection.setDurationInMs(System.currentTimeMillis() - startTimeGeofenceSearch);

            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Returning all geofences, there's room to add them all");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".calculateGeofenceSelection()");

            return selection;
        }
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "There are more than " + MAX_NEAR_GEOFENCES + " geofences, starting selection calculation...");

        // If there are more than 97 recent geofences

        AccuratLogger.log(AccuratLogger.NONE, "Doubling search box");
        // Double starting box
        Double doubledLatitudeRange = startingSearchBox.getLatitudeRange() * 2;
        if (doubledLatitudeRange > MAX_META_GEOFENCE_LAT_RANGE) {
            doubledLatitudeRange = MAX_META_GEOFENCE_LAT_RANGE;
        }
        Double doubledLongitudeRange = startingSearchBox.getLongitudeRange() * 2;
        if (doubledLongitudeRange > MAX_META_GEOFENCE_LNG_RANGE) {
            doubledLongitudeRange = MAX_META_GEOFENCE_LNG_RANGE;
        }
        // Note: the above code should be a method in AccuratGeofenceRange, the MAX_RANGEs should be constants, possibly server-set

        AccuratGeofenceRange doubledStartingBox = new AccuratGeofenceRange(doubledLatitudeRange, doubledLongitudeRange);
        GeofenceSelection selection = getClosestGeofences(locationUpdateWrapper.getContext(), locationUpdateWrapper.getLocation(), doubledStartingBox, 1);
        selection.setDurationInMs(System.currentTimeMillis() - startTimeGeofenceSearch);

        // Store the box where we found enough results
        startingSearchBox = selection.getBox();
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Selection calculated");
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".calculateGeofenceSelection()");

        return selection;
    }

    /**
     * For debugging purposes only.
     * This returns an array of coordinates which can be used to highlight the search area on a map.
     *
     * @return Coordinates from north-west to south-west.
     */
    public static LatLng[] getSearchBox() {
        return searchBox;
    }

    /**
     * For debugging purposes only.
     * This returns all the geofences that are currently being tracked.
     *
     * @return List of all geofences currently being monitored.
     */
    public static List<AccuratGeofence> getCurrentGeofences() {
        return currentMonitoredGeofences;
    }

    /**
     * Monitor the closes geofences.
     *
     * @param context   Needed to interact with the Google API
     * @param selection Selection of geofences to monitor and some meta data about the search
     */
    private static void monitorClosestGeofences(Context context, GeofenceSelection selection) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".monitorClosestGeofences()");
        // verify whether there are geofences to monitor
        if (selection.getGeofences().isEmpty()) {
            AccuratLogger.log(AccuratLogger.GEOFENCE, "Did not update monitoring, selection is empty");

            // update current location geofences
            if (selection.getCurrentLocation() != null) {
                if (CollectionUtils.isEmpty(currentMonitoredGeofences)) {
                    // set new current location meta geofences
                    updateCurrentLocationMetaGeofences(context, selection);
                    AccuratLogger.log(AccuratLogger.GEOFENCE, "Updated current location geofence");
                    AccuratLogger.log(AccuratLogger.NONE, "Now tracking " + currentMonitoredGeofences.size() + " geofences");
                } else {
                    // remove old current location meta geofences
                    geofencingClient.removeGeofences(CURRENT_LOCATION_FENCE_REQUEST_IDS)
                            .addOnSuccessListener(aVoid -> {
                                AccuratLogger.log(AccuratLogger.GEOFENCE, "Stop listening to current location geofence");

                                // remove last geofences in local monitoring cache
                                int targetSize = currentMonitoredGeofences.size() - CURRENT_LOCATION_FENCE_REQUEST_IDS.size();
                                while (currentMonitoredGeofences.size() > targetSize) {
                                    currentMonitoredGeofences.remove(targetSize);
                                }
                                storeCurrentMonitoredGeofences(currentMonitoredGeofences);

                                // set new current location meta geofences
                                updateCurrentLocationMetaGeofences(context, selection);
                                AccuratLogger.log(AccuratLogger.GEOFENCE, "Updated current location geofence");
                                AccuratLogger.log(AccuratLogger.NONE, "Now tracking " + currentMonitoredGeofences.size() + " geofences");
                            })
                            .addOnFailureListener(e -> {
                                AccuratLogger.log(AccuratLogger.ERROR, "Couldn't remove current location geofence: " + e.getMessage());
                            });
                }
            }
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".monitorClosestGeofences()");

            return;
        }

        // Check if the selection is actually different from the currently tracked geofences (currentMonitoredGeofences)
        if (!hasSelectionChanged(selection, currentMonitoredGeofences)) {
            AccuratLogger.log(AccuratLogger.GEOFENCE, "Geofence selection is unchanged, won't restart monitoring");
            AccuratLogger.log(AccuratLogger.NONE, "Still tracking the same " + currentMonitoredGeofences.size() + " geofences");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".monitorClosestGeofences()");

            return;
        }

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Clearing current geofences");
        // clear the current geofences
        clearGeofences(context);

        // track all the fetched geofences
        if (ActivityCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
            // TODO: Consider calling
            //    ActivityCompat#requestPermissions
            // here to request the missing permissions, and then overriding
            //   public void onRequestPermissionsResult(int requestCode, String[] permissions,
            //                                          int[] grantResults)
            // to handle the case where the user grants the permission. See the documentation
            // for ActivityCompat#requestPermissions for more details.
            AccuratLogger.log(AccuratLogger.WARNING, "Can't monitor geofence because " + Manifest.permission.ACCESS_FINE_LOCATION + " has not been granted");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".monitorClosestGeofences()");

            return;
        }

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Going to monitor geofences");
        // monitor the client-defined geofences
        monitorClientGeofences(context, selection);

        // calculate meta geofences
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Going to update meta geofence");
        updateMetaGeofence(context, selection);
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Going to update current location geofence");
        updateCurrentLocationMetaGeofences(context, selection);

        // update logger
        AccuratLogger.log(AccuratLogger.NONE, "Now tracking " + currentMonitoredGeofences.size() + " geofences");
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".monitorClosestGeofences()");
    }

    /**
     * Find out whether a geofence selection is different than the currently monitored geofences.
     *
     * @param selection                 The GeofenceSelection
     * @param currentMonitoredGeofences The list of currently monitored (notification) geofences
     * @return true if the list of geofences differ, false otherwise
     */
    private static boolean hasSelectionChanged(GeofenceSelection selection, List<AccuratGeofence> currentMonitoredGeofences) {
        List<AccuratGeofence> selectionGeofences = selection.getGeofences();
        if (selectionGeofences == null ^ currentMonitoredGeofences == null) {// XOR
            // Only one of both is null
            return true;
        }

        if (currentMonitoredGeofences == null) {
            // Both are null
            return false;
        }

        // Neither is null
        List<AccuratGeofence> currentGeofences = new ArrayList<>(currentMonitoredGeofences.size());
        for (AccuratGeofence geofence : currentMonitoredGeofences) {
            // Filter out the meta and current location geofences
            if (!geofence.getId().equals(META_GEOFENCE_ID) && !geofence.getId().startsWith(CURRENT_LOCATION_GEOFENCE_ID_PREFIX)) {
                currentGeofences.add(geofence);
            }
        }

        return compareGeofenceLists(selectionGeofences, currentGeofences) != 0;
    }

    private static int compareGeofenceLists(@NonNull List<AccuratGeofence> geofences, @NonNull List<AccuratGeofence> otherGeofences) {
        if (geofences.size() != otherGeofences.size()) {
            AccuratLogger.log(AccuratLogger.GEOFENCE, "Geofence lists differ in size: " + geofences.size() + " and " + otherGeofences.size());

            return geofences.size() - otherGeofences.size();
        }

        return checksum(geofences) - checksum(otherGeofences);
    }

    private static int checksum(List<AccuratGeofence> geofences) {
        int checksum = 0;
        for (AccuratGeofence geofence : geofences) {
            checksum += (geofence.getId() + geofence.getLatitude() + geofence.getLongitude() + geofence.getRadius()).hashCode();
        }

        return checksum;
    }


    private static void monitorClientGeofences(Context context, GeofenceSelection selection) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".monitorClientGeofences()");

        // add selected geofences
        currentMonitoredGeofences = selection.getGeofences();
        storeCurrentMonitoredGeofences(currentMonitoredGeofences);
        // create geofencing request for general geofences
        GeofencingRequest.Builder generalGeofencingRequestBuilder = new GeofencingRequest.Builder()
                .setInitialTrigger(GeofencingRequest.INITIAL_TRIGGER_ENTER | GeofencingRequest.INITIAL_TRIGGER_DWELL);
        int geofenceTransition = Geofence.GEOFENCE_TRANSITION_ENTER | Geofence.GEOFENCE_TRANSITION_DWELL | Geofence.GEOFENCE_TRANSITION_EXIT;
        for (AccuratGeofence targetFence : currentMonitoredGeofences) {
            Geofence tranformedGeofence = transformGeofence(targetFence, geofenceTransition);
            if (tranformedGeofence != null) {
                generalGeofencingRequestBuilder.addGeofence(tranformedGeofence);
            }
        }

        // monitor geofences
        monitorGeofences(context, generalGeofencingRequestBuilder.build(), "client");
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".monitorClientGeofences()");
    }

    private static void updateMetaGeofence(Context context, GeofenceSelection selection) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".updateMetaGeofence()");
        // create new meta geofence if there is a need for one
        // (when a filter has been applied to the geofences)
        if (selection.getNumIterations() > 0) {
            Double radius = selection.getBox().convertToDistance();
            AccuratLogger.log(AccuratLogger.GEOFENCE, "Meta geofence radius = " + radius);
            LocationInfo locationInfo = selection.getCurrentLocation().getLocationInfo();
            metaGeofence = new AccuratGeofence(META_GEOFENCE_ID, locationInfo.getLatitude(), locationInfo.getLongitude(), "", new Double[0][0], radius.intValue());
            storeMetaGeofence(metaGeofence);
            if (currentMonitoredGeofences == null) {
                currentMonitoredGeofences = new ArrayList<>();
            }
            currentMonitoredGeofences.add(metaGeofence);
            storeCurrentMonitoredGeofences(currentMonitoredGeofences);

            searchBox = new LatLng[]{
                    new LatLng(locationInfo.getLatitude() - selection.getBox().getLatitudeRange(), locationInfo.getLongitude() + selection.getBox().getLongitudeRange()), // NW
                    new LatLng(locationInfo.getLatitude() + selection.getBox().getLatitudeRange(), locationInfo.getLongitude() + selection.getBox().getLongitudeRange()), // NE
                    new LatLng(locationInfo.getLatitude() + selection.getBox().getLatitudeRange(), locationInfo.getLongitude() - selection.getBox().getLongitudeRange()), // SE
                    new LatLng(locationInfo.getLatitude() - selection.getBox().getLatitudeRange(), locationInfo.getLongitude() - selection.getBox().getLongitudeRange())  // SW
            };

            // monitor meta geofence
            Geofence transformedGeofence = transformGeofence(metaGeofence, Geofence.GEOFENCE_TRANSITION_EXIT);
            if (transformedGeofence != null) {
                GeofencingRequest metaGeofencingRequest = new GeofencingRequest.Builder()
                        .setInitialTrigger(GeofencingRequest.INITIAL_TRIGGER_EXIT)
                        .addGeofence(transformedGeofence)
                        .build();
                monitorGeofences(context, metaGeofencingRequest, "meta");
            } else {
                AccuratLogger.log(AccuratLogger.WARNING, "Can't monitor meta geofence since it is invalid");
            }
        } else {
            AccuratLogger.log(AccuratLogger.NONE, "No need for a meta geofence, because the number of geofences is < " + MAX_NEAR_GEOFENCES);
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".updateMetaGeofence()");
    }

    private static void updateCurrentLocationMetaGeofences(Context context, GeofenceSelection selection) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".updateCurrentLocationMetaGeofences()");
        // create geofencing request for current location meta geofences
        GeofencingRequest.Builder metaGeofencingRequestBuilder = new GeofencingRequest.Builder()
                .setInitialTrigger(0);

        AccuratSettingsManager.init(context);
        // add location-improving meta geofences to monitored list
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Setting radius for current location geofence to " + AccuratSettingsManager.getSettings().getCurrentGeofenceRadius() + " meter");
        AccuratGeofence metaFence = new AccuratGeofence(
                CURRENT_LOCATION_FENCE_REQUEST_IDS.get(0),
                selection.getCurrentLocation().getLocationInfo().getLatitude(),
                selection.getCurrentLocation().getLocationInfo().getLongitude(),
                null,
                new Double[0][0],
                AccuratSettingsManager.getSettings().getCurrentGeofenceRadius()
        );
        if (currentMonitoredGeofences == null) {
            currentMonitoredGeofences = new ArrayList<>();
        }
        currentMonitoredGeofences.add(metaFence);
        storeCurrentMonitoredGeofences(currentMonitoredGeofences);
        Geofence transformedGeofence = transformGeofence(metaFence, Geofence.GEOFENCE_TRANSITION_EXIT);
        if (transformedGeofence != null) {
            metaGeofencingRequestBuilder.addGeofence(transformedGeofence);
            // monitor meta geofences
            monitorGeofences(context, metaGeofencingRequestBuilder.build(), "current location");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".updateCurrentLocationMetaGeofences()");
        } else {
            AccuratLogger.log(AccuratLogger.WARNING, "Can't monitor current location geofence since it is invalid");
        }
    }

    /**
     * Transform our custom Geofence format to something Google Location Services understands
     *
     * @param geofence               Geofence to transform
     * @param geofenceTransitionType Type of transition, use {@link Geofence#GEOFENCE_TRANSITION_ENTER} and the like
     * @return Geofence to monitor
     */
    private static Geofence transformGeofence(AccuratGeofence geofence, int geofenceTransitionType) {
        try {
            return new Geofence.Builder()
                    .setRequestId(geofence.getId())
                    .setCircularRegion(geofence.getLatitude(), geofence.getLongitude(), geofence.getRadius())
                    .setExpirationDuration(GEOFENCE_EXPIRATION_IN_MS)
                    .setTransitionTypes(geofenceTransitionType)
                    .setLoiteringDelay(GEOFENCE_LOITERING_DELAY)
                    .build();
        } catch (IllegalArgumentException e) {
            AccuratLogger.log(AccuratLogger.ERROR, e.getMessage());

            return null;
        }
    }

    /**
     * Actually start monitoring geofences by adding them to the geofencingclient.
     *
     * @param context           Context used to create the {@link PendingIntent} which'll be executed when a geofence in this request is triggered.
     * @param geofencingRequest Request to pass to the geofencing client. Contains geofences and trigger and transition types.
     * @param geofenceType      Identifies the kind of geofences to monitor, used for logging purposes.
     */
    @SuppressLint("MissingPermission")
    private static void monitorGeofences(Context context, GeofencingRequest geofencingRequest, String geofenceType) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".monitorGeofences()");
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Adding " + geofencingRequest.getGeofences().size() + " geofences (" + geofenceType + ") to geofencingClient");
        geofencingClient.addGeofences(geofencingRequest, getGeofencePendingIntent(context))
                .addOnSuccessListener(aVoid -> {
                    AccuratLogger.log(AccuratLogger.SDK_FLOW, "Successfully monitoring " + geofenceType + " geofences");
                })
                .addOnFailureListener(e -> {
                    AccuratLogger.log(AccuratLogger.ERROR, "Failed to start monitoring " + geofenceType + " geofences: " + e.getMessage());
                });
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".monitorGeofences()");
    }

    private static void clearGeofences(Context context) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".clearGeofences()");
        geofencingClient.removeGeofences(getGeofencePendingIntent(context))
                .addOnSuccessListener(aVoid -> {
                    AccuratLogger.log(AccuratLogger.SDK_FLOW, "Successfully stopped listening to geofences");
                })
                .addOnFailureListener(e -> {
                    AccuratLogger.log(AccuratLogger.ERROR, "Could not remove geofences: " + e.getMessage());
                });
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".clearGeofences()");
    }

    private static PendingIntent getGeofencePendingIntent(final Context context) {
        // Reuse the PendingIntent if we already have it.
        if (geofencePendingIntent != null) {
            return geofencePendingIntent;
        }
        Intent intent = new Intent(context, GeofenceBroadcastReceiver.class);
        // We use FLAG_UPDATE_CURRENT so that we get the same pending intent back when
        // calling addGeofences() and removeGeofences().
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            geofencePendingIntent = PendingIntent.getBroadcast(context, 0, intent, PendingIntent.FLAG_UPDATE_CURRENT | PendingIntent.FLAG_IMMUTABLE);
        } else {
            geofencePendingIntent = PendingIntent.getBroadcast(context, 0, intent, PendingIntent.FLAG_UPDATE_CURRENT);
        }

        return geofencePendingIntent;
    }
    //</editor-fold>
    //</editor-fold>

    //<editor-fold desc="API calls">

    /**
     * Checks whether the geofences have been updated since the last time
     *
     * @param context  Used to check network state and access the storage
     * @param callback Used to communicate success and actions (new geofences have been added to storage)
     */
    private static void fetchStatus(final Context context, @NonNull final AccuratActionableCallback callback) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".fetchStatus()");
        if (!AccuratApi.isNetworkAvailable(context)) {
            AccuratLogger.log(AccuratLogger.NETWORK, "Network not available");
            fail(callback);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".fetchStatus()");

            return;
        }

        // Build geofences state request
        JSONObject requestBody = new JSONObject();
        try {
            AccuratLogger.log(AccuratLogger.STORAGE, "Local geofence checksum = " + mStorage.getString(StorageKeys.ACCURAT_GEOFENCES_STATUS, "unset"));
            requestBody.put(JSON_GEOFENCE_CHECKSUM, mStorage.getString(StorageKeys.ACCURAT_GEOFENCES_STATUS, "unknown"));
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to load local geofence checksum from storage: " + e.getMessage());
            e.printStackTrace();
        }
        JsonObjectRequest statusRequest = new JsonObjectRequest(
                Request.Method.POST,
                AccuratEndpoints.GET_GEOFENCES_STATE.getUrl(),
                requestBody,
                response -> {
                    AccuratLogger.logNetworkResponse(HttpMethod.POST, Configuration.ENDPOINT_GET_GEOFENCES_STATE, response, false);
                    handleGeofencesState(context, response, callback);
                },
                error -> {
                    AccuratLogger.logNetworkError(HttpMethod.POST, Configuration.ENDPOINT_GET_GEOFENCES_STATE, error);
                    fail(callback);
                }) {
            @Override
            public Map<String, String> getHeaders() throws AuthFailureError {
                return AccuratApi.getHeaders(mStorage, "POST", getBodyContentType(),
                        AccuratApi.getEncodedRequestBody(requestBody.toString()),
                        AccuratEndpoints.GET_GEOFENCES_STATE.getPath());
            }
        };

        statusRequest.setTag(TAG)
                .setRetryPolicy(AccuratApi.defaultRetryPolicy)
                .setShouldCache(false);

        // Request geofences file state
        if (requestQueue == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "requestQueue is NULL, re-initialising queue....");
            requestQueue = Volley.newRequestQueue(context);
        }

        AccuratLogger.logNetworkRequest(HttpMethod.POST, Configuration.ENDPOINT_GET_GEOFENCES_STATE, requestBody, false);
        requestQueue.add(statusRequest);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".fetchStatus()");
    }

    /**
     * Fetches the geofences.
     *
     * @param context  Used for network and storage access.
     * @param callback Used to communicate success, failure and potential actions (when storage has been updated)
     */
    private static void fetchGeofences(final Context context, @NonNull final AccuratActionableCallback callback) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".fetchGeofences()");
        if (!AccuratSettingsManager.isGeofencingEnabled()) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Not fetching geofences: Geofencing is disabled");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".fetchGeofences()");
            callback.onCompleted(true);

            return;
        }

        if (!AccuratApi.isNetworkAvailable(context)) {
            AccuratLogger.log(AccuratLogger.NETWORK, "Network not available");
            fail(callback);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".fetchGeofences()");

            return;
        }

        // Build geofences request
        JsonArrayRequest geofencesRequest = new JsonArrayRequest(
                Request.Method.GET,
                AccuratEndpoints.GET_GEOFENCES.getUrl(),
                null,
                response -> {
                    AccuratLogger.logNetworkResponse(HttpMethod.GET, Configuration.ENDPOINT_GET_GEOFENCES, response, false);
                    storeGeofences(context, response, callback);
                },
                error -> {
                    AccuratLogger.logNetworkError(HttpMethod.GET, Configuration.ENDPOINT_GET_GEOFENCES, error);
                    fail(callback);
                }) {
            @Override
            public Map<String, String> getHeaders() throws AuthFailureError {
                return AccuratApi.getHeaders(mStorage, "GET", "",
                        AccuratApi.getEncodedRequestBody(""), AccuratEndpoints.GET_GEOFENCES.getPath());
            }
        };

        geofencesRequest.setTag(TAG)
                .setRetryPolicy(AccuratApi.defaultRetryPolicy)
                .setShouldCache(false);

        // Request geofences
        if (requestQueue == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "requestQueue is NULL, re-initialising queue....");
            AccuratLogger.log(AccuratLogger.ERROR, "Failed to make API call, requestQueue is null");
            requestQueue = Volley.newRequestQueue(context);
        }

        AccuratLogger.logNetworkRequest(HttpMethod.GET, Configuration.ENDPOINT_GET_GEOFENCES);
        requestQueue.add(geofencesRequest);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".fetchGeofence()");
    }

    //<editor-fold desc="Notifications">

    /**
     * Show potential notifications for a geofence trigger.
     *
     * @param context            Used for network access.
     * @param geofence           The geofence that has been triggered.
     * @param completionCallback Callback which indicates whether the request has been posted correctly.
     */
    public static void showNotificationForGeofence(final Context context, @NonNull final AccuratGeofence geofence, @NonNull final AccuratCompletionCallback completionCallback) {
        showNotificationForGeofence(context, geofence, false, completionCallback);
    }

    /**
     * Show potential notifications for a geofence trigger.
     *
     * @param context            Used for network access.
     * @param geofence           The geofence that has been triggered.
     * @param isRetry            Whether the geofence had been triggered before but network was unavailable, and if the current call is another attempt.
     * @param completionCallback Callback which indicates whether the request has been posted correctly.
     */
    public static void showNotificationForGeofence(final Context context, @NonNull final AccuratGeofence geofence, final boolean isRetry, @NonNull final AccuratCompletionCallback completionCallback) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".showNotificationForGeofence()");
        RealmManager.init(context);
        // Only POST geofences when network is fully functional
        if (!AccuratApi.isNetworkAvailable(context)) {
            AccuratLogger.log(AccuratLogger.NETWORK, "No network available, can't fetch online notification for geofence with ID " + geofence.getId());

            showOfflineNotification(context, geofence, completionCallback);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".showNotificationForGeofence()");

            return;
        }

        // An AD ID is required to request a possible notification when a certain user is triggering a geofence
        String adId;
        try {
            adId = AdvertisingManager.getAdId();
        } catch (IllegalStateException e) {
            AccuratLogger.log(AccuratLogger.ERROR, "AdvertisingManager not initialised: " + e.getMessage());
            AdvertisingManager.init(context);
            adId = AdvertisingManager.getAdId();
        }

        if (TextUtils.isEmpty(adId) || AdvertisingManager.isAdTrackingLimited()) {
            AccuratLogger.log(AccuratLogger.WARNING, "No valid ad ID, won't fetch online notification");
            completionCallback.onCompleted(false);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".showNotificationForGeofence()");

            return;
        }

        // Show notification for first notificationId which hasn't exceeded the frequency limit yet
        Long[] notificationIds = geofence.getExtractedNotificationIds();
        if (notificationIds == null || notificationIds.length == 0) {
            // No notifications attached to geofence, nothing to do.
            AccuratLogger.log(AccuratLogger.NOTIFICATION, "No notifications for geofence");
            complete(completionCallback, true);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".showNotificationForGeofence()");

            return;
        }

        LongSparseArray<GeofenceNotification> notifications = GeofenceNotificationManager.getNotifications(notificationIds);
        if (notifications == null || notifications.size() == 0) {
            // No notifications in storage, nothing to do.
            AccuratLogger.log(AccuratLogger.NOTIFICATION, "No notifications in storage for geofence");
            complete(completionCallback, true);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".showNotificationForGeofence()");

            return;
        }

        if(!shouldFetchOnlineNotification(context, new Date(), notificationIds, notifications)) {
            AccuratLogger.log(AccuratLogger.NOTIFICATION, "No suitable notification for showing, won't fetch online");
            complete(completionCallback, false);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".showNotificationForGeofence()");

            return;
        }

        // Init managers
        AccuratUserManager.init(context);
        init(context); // ensure there's an active requestQueue to send these requests to

        // Build request
        final HashMap<String, Object> urlParameters = new HashMap<>();
        urlParameters.put(ApiKeys.Url.AD_ID, adId);
        JSONObject requestBody = getRequestBodyForGeofenceTrigger(geofence);
        JsonObjectRequest getNotificationRequest = new JsonObjectRequest(
                Request.Method.POST,
                AccuratEndpoints.POST_GEOFENCE_TRIGGER.getUrl(urlParameters),
                requestBody,
                response -> {
                    AccuratLogger.logNetworkResponse(HttpMethod.POST, Configuration.ENDPOINT_POST_GEOFENCE_TRIGGER, response, false);
                    processGeofenceTriggerResponse(context, response, completionCallback);
                },
                error -> {
                    AccuratLogger.logNetworkError(HttpMethod.POST, Configuration.ENDPOINT_POST_GEOFENCE_TRIGGER, error);
                    completionCallback.onCompleted(false);
                }
        ) {
            @Override
            public Map<String, String> getHeaders() {
                return AccuratApi.getHeaders(mStorage, "POST", getBodyContentType(),
                        AccuratApi.getEncodedRequestBody(requestBody.toString()),
                        AccuratEndpoints.POST_GEOFENCE_TRIGGER.getPath(urlParameters));
            }
        };
        getNotificationRequest.setRetryPolicy(new DefaultRetryPolicy(20000, 1, DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));

        // Perform the request
        AccuratLogger.logNetworkRequest(HttpMethod.POST, Configuration.ENDPOINT_POST_GEOFENCE_TRIGGER, requestBody, false);
        requestQueue.add(getNotificationRequest);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".showNotificationForGeofence()");
    }

    /**
     * Compose the request body when a geofence has been triggered.
     *
     * @param geofence The triggered geofence.
     * @return JSON object to include in the request to {@link AccuratEndpoints#POST_GEOFENCE_TRIGGER}.
     */
    private static JSONObject getRequestBodyForGeofenceTrigger(@NonNull AccuratGeofence geofence) {
        JSONObject requestBody = new JSONObject();

        try {
            // Check whether there are nids for this geofence
            if (TextUtils.isEmpty(geofence.getNids())) {
                requestBody.put(JSON_ROOT, null);
            } else {
                AccuratLanguage userLanguage = AccuratUserManager.getUser().getLanguage();
                if (userLanguage != null) {
                    requestBody.put(POST_BODY_PARAM_LANGUAGE, userLanguage.getCode());
                }
                requestBody.put(POST_BODY_PARAM_NIDS, geofence.getNids());
                requestBody.put(POST_BODY_PARAM_ID, geofence.getId());
            }
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".getRequestBodyForGeofenceTrigger(): " + e.getMessage());
            e.printStackTrace();
        }

        return requestBody;
    }

    /**
     * Processes the response received when posting the triggered geofence.
     * Shows a notification if need be.
     *
     * @param context            Context used to compose the notification.
     * @param response           Response received after calling {@link AccuratEndpoints#POST_GEOFENCE_TRIGGER}.
     * @param completionCallback Callback will indicate whether the response was valid and was used to show a notification.
     */
    private static void processGeofenceTriggerResponse(Context context, JSONObject response, AccuratCompletionCallback completionCallback) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".processGeofenceTriggerResponse()");
        AccuratLogger.init(context);
        if (response == null || response.isNull(JSON_ROOT)) {
            // Might be NULL if the notification was already triggered earlier today
            AccuratLogger.log(AccuratLogger.WARNING, "Could not process geofence notification, response is NULL (eg. actual error or when notification was already shown earlier) or missing 'data' element");
            complete(completionCallback, false);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".processGeofenceTriggerResponse()");

            return;
        }

        try {
            // Get notification from JSON in the response body
            JSONObject jsonNotification = response.getJSONObject(ServerDataKeys.GeofenceTrigger.DATA);

            // Build the notification
            GeofenceNotification geofenceNotification = GeofenceNotification.fromOnlineJson(jsonNotification);

            // Show the notification
            int notificationSystemId = showNotification(context, geofenceNotification);

            if (notificationSystemId == 0) {
                complete(completionCallback, false);

                return;
            } else {
                GeofenceNotificationLog notificationLog = GeofenceNotificationManager.getNotificationLog();
                notificationLog.addNotification(geofenceNotification.getNotificationId(), new Date());
                GeofenceNotificationManager.storeNotificationLog();
            }

            AccuratLogger.log(AccuratLogger.NOTIFICATION, "NOTIFICATION - Showing notification with Accurat id "
                    + geofenceNotification.getNotificationId() + " [ "
                    + geofenceNotification.getTitle() + " | "
                    + geofenceNotification.getMessage() + " ]");
            AccuratLogger.log(AccuratLogger.NONE, "System notification ID = " + notificationSystemId);

            complete(completionCallback, true);
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to parse notification: " + e.getMessage());
            e.printStackTrace();
            complete(completionCallback, false);
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".processGeofenceTriggerResponse()");
    }

    /**
     * Show a GeofenceNotification to the user.
     *
     * @param context              The {@link Context}
     * @param geofenceNotification The {@link GeofenceNotification}
     * @return int The system notification ID
     */
    private static int showNotification(Context context, GeofenceNotification geofenceNotification) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".showNotification()");
        if (mStorage == null) {
            AccuratLogger.log(AccuratLogger.NOTIFICATION, "Can't showNotification, storage not initialised");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".showNotification()");

            return 0;
        }
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Showing notification " + geofenceNotification);

        int notificationSystemId = mStorage.getInt(StorageKeys.ACCURAT_LAST_NOTIFICATION_ID, 2335) + 1;
        if (notificationSystemId == 0) {
            notificationSystemId++;
        }

        int notificationColor;
        if(geofenceNotification == null || geofenceNotification.getBrandColour() == null) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                notificationColor = context.getResources().getColor(R.color.accurat_default_notification_color, null);
            } else {
                notificationColor = context.getResources().getColor(R.color.accurat_default_notification_color);
            }
        } else {
            notificationColor = Color.parseColor(geofenceNotification.getBrandColour());
        }

        Notification notification = new NotificationCompat.Builder(context, notificationChannelId)
                .setAutoCancel(true)
                .setContentTitle(geofenceNotification.getTitle())
                .setContentText(geofenceNotification.getMessage())
                .setDefaults(NotificationCompat.DEFAULT_SOUND)
                .setSmallIcon(getNotificationIcon())
                .setColor(notificationColor)
                .setCategory(NotificationCompat.CATEGORY_PROMO)
                .setGroup(notificationChannelId)
                .setContentIntent(createNotificationAction(context, String.valueOf(geofenceNotification.getNotificationId()), geofenceNotification.getData()))
                .build();

        NotificationManagerCompat notificationManager = NotificationManagerCompat.from(context);
        notificationManager.notify(notificationSystemId, notification);

        mStorage.setValue(StorageKeys.ACCURAT_LAST_NOTIFICATION_ID, notificationSystemId)
                .commit();
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".showNotification()");

        return notificationSystemId;
    }

    /**
     * Show an offline notification.
     *
     * @param context            The {@link Context}
     * @param geofence           The {@link AccuratGeofence} to show a notification for
     * @param completionCallback An optional {@link AccuratCompletionCallback}
     */
    private static void showOfflineNotification(Context context, AccuratGeofence geofence, AccuratCompletionCallback completionCallback) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".showOfflineNotification()");
        if (context == null) {
            // Invalid parameters, can't do anything.
            AccuratLogger.log(AccuratLogger.NOTIFICATION, "Can't showOfflineNotification, context is NULL");
            complete(completionCallback, false);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".showOfflineNotification()");

            return;
        }
        init(context);

        if (geofence == null) {
            // Invalid parameters, can't do anything.
            AccuratLogger.log(AccuratLogger.NOTIFICATION, "Can't showOfflineNotification, geofence is NULL");
            complete(completionCallback, false);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".showOfflineNotification()");

            return;
        }

        // Show notification for first notificationId which hasn't exceeded the frequency limit yet
        Long[] notificationIds = geofence.getExtractedNotificationIds();
        if (notificationIds == null || notificationIds.length == 0) {
            // No notifications attached to geofence, nothing to do.
            AccuratLogger.log(AccuratLogger.NOTIFICATION, "No notifications for geofence");
            complete(completionCallback, true);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".showOfflineNotification()");

            return;
        }

        LongSparseArray<GeofenceNotification> notifications = GeofenceNotificationManager.getNotifications(notificationIds);
        if (notifications == null || notifications.size() == 0) {
            // No notifications in storage, nothing to do.
            AccuratLogger.log(AccuratLogger.NOTIFICATION, "No notifications in storage for geofence");
            complete(completionCallback, true);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".showOfflineNotification()");

            return;
        }

        Date now = new Date();
        GeofenceNotification notification = findSuitableNotification(context, now, notificationIds, notifications);
        if (notification == null) {
            // No suitable notification found, nothing to do.
            AccuratLogger.log(AccuratLogger.NOTIFICATION, "No suitable notification found in storage for geofence");
            complete(completionCallback, true);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".showOfflineNotification()");

            return;
        }

        int systemId = showNotification(context, notification);

        if(systemId != 0) {
            // Add the notification to the notification log
            GeofenceNotificationLog notificationLog = GeofenceNotificationManager.getNotificationLog();
            notificationLog.addNotification(notification.getNotificationId(), now);
            GeofenceNotificationManager.storeNotificationLog();

            // Add an interaction for the triggered offline notification
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Adding interact for offline notification with nid " + notification.getNotificationId());
            boolean success = CampaignManager.addOfflineNotificationInteraction(notification.getNotificationId());
            if (success) {
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Successfully added interaction");
            } else {
                AccuratLogger.log(AccuratLogger.WARNING, "Failed to add interaction");
            }

            complete(completionCallback, true);
        } else {
            complete(completionCallback, false);
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".showOfflineNotification()");
    }

    private static boolean shouldFetchOnlineNotification(@NonNull Context context, @NonNull Date now, @NonNull Long[] notificationIds, @NonNull LongSparseArray<GeofenceNotification> notifications) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".shouldFetchOnlineNotification()");
        AccuratSettingsManager.init(context);
        GeofenceNotificationManager.init(context);
        int maxNotificationsPerDay = AccuratSettingsManager.getSettings().getMaxNotificationsPerDay();
        GeofenceNotificationLog notificationLog = GeofenceNotificationManager.getNotificationLog();

        if (notificationLog == null) {
            AccuratLogger.log(AccuratLogger.NONE, "No notification log in storage, returning false");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".shouldFetchOnlineNotification()");

            return false;
        }

        AccuratLogger.log(AccuratLogger.NONE, "Notification log = " + notificationLog.toString());

        int notificationCountToday = notificationLog.getNotificationCountForDate(now);
        if (notificationCountToday >= maxNotificationsPerDay) {
            // Maximum notification count for today has been reached, don't fetch a notification.
            AccuratLogger.log(AccuratLogger.NOTIFICATION, "User has reached the daily notification limit");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".shouldFetchOnlineNotification()");

            return false;
        }

        for (long notificationId : notificationIds) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Checking notification ID " + notificationId);
            if (notifications.indexOfKey(notificationId) < 0) {
                // There is no notification in storage for the notificationId, skip to the next notificationId
                AccuratLogger.log(AccuratLogger.NONE, "No notification in storage for ID " + notificationId);
                continue;
            }

            GeofenceNotification notification = notifications.get(notificationId);
            if (!notification.canShowAt(now)) {
                // The notification timings don't fit now, skip to the next notificationId
                AccuratLogger.log(AccuratLogger.NONE, "Timing " + GeofenceNotificationLog.TIME_FORMAT.format(now) + " doesn't fit notification timings: " + notification.toString());
                continue;
            }

            // Check the frequency of the notification
            String previousNotificationDate = notificationLog.getMostRecentDateForNotificationId(notificationId, now);
            AccuratLogger.log(AccuratLogger.NONE, "Previous notification date = " + previousNotificationDate);
            if (previousNotificationDate == null) {
                // The notification has never been shown before, so use this one
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Found suitable notification: " + notification);
                AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".shouldFetchOnlineNotification()");

                return true;
            }

            if (notification.getFrequency() <= 0) {
                // The notification should only be shown once, and has already been shown, skip to the next notificationId
                AccuratLogger.log(AccuratLogger.NONE, "Notification already shown once [frequency = " + notification.getFrequency() + "]");
                continue;
            }

            long dayDifference = getDayDifference(now, previousNotificationDate);
            AccuratLogger.log(AccuratLogger.NONE, "dayDifference = " + dayDifference);
            if (dayDifference < notification.getFrequency()) {
                // The notification was shown less than 'frequency' days ago, skip to the next notificationId
                AccuratLogger.log(AccuratLogger.NONE, "Notification already shown too recently [frequency = " + notification.getFrequency() + ", shown " + dayDifference + " days ago]");
                continue;
            }

            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Found suitable notification: " + notification);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".shouldFetchOnlineNotification()");

            return true;
        }

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "No suitable notification found");
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".shouldFetchOnlineNotification()");

        return false;
    }

    private static GeofenceNotification findSuitableNotification(@NonNull Context context, @NonNull Date now, @NonNull Long[] notificationIds, @NonNull LongSparseArray<GeofenceNotification> notifications) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".findSuitableNotification()");
        AccuratSettingsManager.init(context);
        GeofenceNotificationManager.init(context);
        int maxNotificationsPerDay = AccuratSettingsManager.getSettings().getMaxNotificationsPerDay();
        GeofenceNotificationLog notificationLog = GeofenceNotificationManager.getNotificationLog();

        if (notificationLog == null) {
            AccuratLogger.log(AccuratLogger.NONE, "No notification log in storage, creating a new one");
            notificationLog = new GeofenceNotificationLog();
        }

        AccuratLogger.log(AccuratLogger.NONE, "Notification log = " + notificationLog.toString());

        int notificationCountToday = notificationLog.getNotificationCountForDate(now);
        if (notificationCountToday >= maxNotificationsPerDay) {
            // Maximum notification count for today has been reached, don't return a notification.
            AccuratLogger.log(AccuratLogger.NOTIFICATION, "User has reached the daily notification limit");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".findSuitableNotification()");

            return null;
        }

        for (long notificationId : notificationIds) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Checking notification ID " + notificationId);
            if (notifications.indexOfKey(notificationId) < 0) {
                // There is no notification in storage for the notificationId, skip to the next notificationId
                AccuratLogger.log(AccuratLogger.NONE, "No notification in storage for ID " + notificationId);
                continue;
            }

            GeofenceNotification notification = notifications.get(notificationId);
            if (!notification.canShowAt(now)) {
                // The notification timings don't fit now, skip to the next notificationId
                AccuratLogger.log(AccuratLogger.NONE, "Timing " + GeofenceNotificationLog.TIME_FORMAT.format(now) + " doesn't fit notification timings: " + notification.toString());
                continue;
            }

            // Check the frequency of the notification
            String previousNotificationDate = notificationLog.getMostRecentDateForNotificationId(notificationId, now);
            AccuratLogger.log(AccuratLogger.NONE, "Previous notification date = " + previousNotificationDate);
            if (previousNotificationDate == null) {
                // The notification has never been shown before, so use this one
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Found suitable notification: " + notification);
                AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".findSuitableNotification()");

                return notification;
            }

            if (notification.getFrequency() <= 0) {
                // The notification should only be shown once, and has already been shown, skip to the next notificationId
                AccuratLogger.log(AccuratLogger.NONE, "Notification already shown once [frequency = " + notification.getFrequency() + "]");
                continue;
            }

            long dayDifference = getDayDifference(now, previousNotificationDate);
            AccuratLogger.log(AccuratLogger.NONE, "dayDifference = " + dayDifference);
            if (dayDifference < notification.getFrequency()) {
                // The notification was shown less than 'frequency' days ago, skip to the next notificationId
                AccuratLogger.log(AccuratLogger.NONE, "Notification already shown too recently [frequency = " + notification.getFrequency() + ", shown " + dayDifference + " days ago]");
                continue;
            }

            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Found suitable notification: " + notification);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".findSuitableNotification()");

            return notification;
        }

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "No suitable notification found");
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".findSuitableNotification()");

        return null;
    }

    /**
     * Get custom notification icon
     *
     * @return A custom notification icon configured by the client, or a fallback icon.
     */
    private static int getNotificationIcon() {
        return mStorage.getInt(StorageKeys.CLIENT_NOTIFICATION_DRAWABLE_ID, R.mipmap.ic_accurat_notification);
    }

    /**
     * Creates a {@link PendingIntent} which should be launched when the notification is clicked.
     *
     * @param context          Context to use for intent creation
     * @param notificationId   Notification ID to include in the intent
     * @param notificationData Stringyfied JSON Object representing data to pass to the hosting application
     * @return Pending Intent which could be used to open the implementing app, or null when no target class has been configured
     */
    private static PendingIntent createNotificationAction(Context context, String notificationId, String notificationData) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".createNotificationAction()");
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Creating action for notification ID " + notificationId);

        // Retrieve target from local storage
        String targetClass = AccuratConfigurationManager.getNotificationTargetClass();
        String packageName = AccuratConfigurationManager.getNotificationTargetPackage();
        AccuratLogger.log(AccuratLogger.NONE, "targetClass = " + targetClass + " | packageName = " + packageName);
        if (TextUtils.isEmpty(packageName)) {
            AccuratLogger.log(AccuratLogger.ERROR, "Can't create notification action, packageName is empty");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".createNotificationAction()");

            return null;
        }

        // Compose intent that'll be used to launch the client
        PackageManager pm = context.getPackageManager();
        Intent intent;
        if (TextUtils.isEmpty(targetClass)) {
            // Use launch intent as fallback
            AccuratLogger.log(AccuratLogger.WARNING, "targetClass is empty, trying to use launch intent");
            intent = pm.getLaunchIntentForPackage(packageName);
            if (intent == null) {
                AccuratLogger.log(AccuratLogger.ERROR, "Can't create notification action, targetClass is empty and launchIntent is NULL");
                AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".createNotificationAction()");

                return null;
            }
        } else {
            // Compose custom launch intent
            intent = new Intent();
            intent.setClassName(packageName, targetClass);
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Successfully created PendingIntent for notification action");
        }
        addNotificationExtras(notificationId, intent, notificationData);

        // Verify whether the composed intent is supported by any activity on the device
        List<ResolveInfo> activities = pm.queryIntentActivities(intent, PackageManager.MATCH_DEFAULT_ONLY);
        boolean isIntentSafe = activities.size() > 0;
        if (isIntentSafe) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "PendingIntent is usable");
        } else {
            // Use launch intent as fallback
            AccuratLogger.log(AccuratLogger.WARNING, "PendingIntent is unusable, trying to use launch intent");
            intent = pm.getLaunchIntentForPackage(packageName);
            if (intent == null) {
                AccuratLogger.log(AccuratLogger.ERROR, "No launch intent found for package " + packageName);
                AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".createNotificationAction()");

                return null;
            }
            AccuratLogger.log(AccuratLogger.NONE, "Launch intent is usable");
            addNotificationExtras(notificationId, intent, notificationData);
        }

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".createNotificationAction()");

        // Create pending intent for notification
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            return PendingIntent.getActivity(context, RequestCodes.NOTIFICATION_REQUEST_CODE, intent, PendingIntent.FLAG_IMMUTABLE);
        } else {
            return PendingIntent.getActivity(context, RequestCodes.NOTIFICATION_REQUEST_CODE, intent, 0);
        }
    }

    /**
     * Add parameters to the geofence notification action intent.
     *
     * @param notificationId   Notification ID to pass to the client.
     * @param intent           Target intent to add parameters to -- will be launched by the system when the notification is tapped.
     * @param notificationData Stringyfied JSON Object representing data to pass to the hosting application
     */
    private static void addNotificationExtras(String notificationId, Intent intent, String notificationData) {
        intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_CLEAR_TASK);
        intent.putExtra(EXTRA_NOTIFICATION_ID, notificationId);
        intent.putExtra(EXTRA_NOTIFICATION_DATA, notificationData);
    }
    //</editor-fold>
    //</editor-fold>

    //<editor-fold desc="Handlers">

    /**
     * Determines the next steps with the state of the geofences file
     *
     * @param context  Context to access storage, and network if need be
     * @param response Response of {@link AccuratEndpoints#GET_GEOFENCES_STATE}
     * @param callback Used to communicate success, failure and potential actions (when storage has been updated)
     */
    private static void handleGeofencesState(Context context, JSONObject response, @NonNull final AccuratActionableCallback callback) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".handleGeofencesState()");
        if (response == null || !response.has(JSON_ROOT)) {
            AccuratLogger.log(AccuratLogger.ERROR, "Can't process geofence state, response is NULL or missing 'data' element");
            fail(callback);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleGeofencesState()");

            return;
        }

        // Verify validity of response
        final JSONObject data;
        try {
            data = response.getJSONObject(JSON_ROOT);
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to parse geofence state: " + e.getMessage());
            fail(callback);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleGeofencesState()");

            return;
        }

        if (data == null || !data.has(JSON_GEOFENCE_STATUS_KEY)) {
            AccuratLogger.log(AccuratLogger.ERROR, "'data' element is NULL or missing 'status' attribute");
            fail(callback);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleGeofencesState()");

            return;
        }

        // Check whether new geofences are available and download them if need be
        try {
            if (JSON_GEOFENCE_STATUS_UPDATED.equals(data.getString(JSON_GEOFENCE_STATUS_KEY))) {
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Geofences have been updated on the server");
                // remove old geofences
                clearGeofences(context);
                removeOldGeofencesFromDatabase(context);
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Going to fetch geofences");
                // get newest geofences
                fetchGeofences(context, new AccuratActionableCallback() {
                    @Override
                    public void onCompleted(boolean success) {
                        // save current hash
                        try {
                            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Setting local geofences checksum to " + data.getString(JSON_GEOFENCE_CHECKSUM));
                            mStorage.setValue(StorageKeys.ACCURAT_GEOFENCES_STATUS, data.getString(JSON_GEOFENCE_CHECKSUM)).commit();
                        } catch (JSONException e) {
                            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to set local geofence checksum: " + e.getMessage());
                        }

                        callback.onCompleted(success);
                    }

                    @Override
                    public void onActionPotentiallyRequired(boolean isActionRequired) {
                        callback.onActionPotentiallyRequired(isActionRequired);
                    }
                });
            } else {
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Geofences have not been updated on the server");
                callback.onCompleted(true);
                callback.onActionPotentiallyRequired(false);
            }
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to parse geofence status: " + e.getMessage());
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".handleGeofencesState()");
    }
    //</editor-fold>

    //<editor-fold desc="Storage">

    /**
     * Stores geofences in local storage
     *
     * @param context  Used to access local storage.
     * @param response The geofences that should be stored.
     * @param callback Callback to indicate that the geofences in local storage have been updated.
     */
    private static void storeGeofences(final Context context, final JSONArray response, @NonNull final AccuratActionableCallback callback) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".storeGeofences()");

        // Parse geofences
        final Gson gson = new Gson();
        final int numGeofences = response.length();
        if (numGeofences == 0) {
            AccuratLogger.log(AccuratLogger.STORAGE, "No geofences to store");
            callback.onCompleted(true);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeGeofences()");

            return;
        }
        AccuratLogger.log(AccuratLogger.DATABASE, "Storing " + numGeofences + " geofences");
        List<AccuratGeofence> geofences = new ArrayList<>(numGeofences);
        for (int i = 0; i < numGeofences; i++) {
            try {
                JSONObject geofenceData = response.getJSONObject(i);
                AccuratGeofence geofence = gson.fromJson(geofenceData.toString(), AccuratGeofence.class);
                geofences.add(geofence);
            } catch (JSONException e) {
                AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to parce geofences response: " + e.getMessage());
            }
        }

        // Store geofences in database
        GeofenceDataSource accuratDatabase;
        try {
            accuratDatabase = new AccuratDatabase(DatabaseHelper.getInstance(context));
            accuratDatabase.addGeofences(geofences);
            AccuratLogger.log(AccuratLogger.DATABASE, "Stored " + geofences.size() + " geofences");
        } catch (Exception e) {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not add geofences to database: " + e.getMessage());
        }

        // Report number of fences in database
        AccuratDatabase countDatabase;
        try {
            countDatabase = new AccuratDatabase(DatabaseHelper.getInstance(context));
            long numStoredGeofences = countDatabase.getNumberOfGeofences();
            AccuratLogger.log(AccuratLogger.DATABASE, "Database now has " + numStoredGeofences + " geofences");
        } catch (Exception e) {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not get number of stored geofences: " + e.getMessage());
        }

        // Report result to callback
        callback.onCompleted(true);
        callback.onActionPotentiallyRequired(false);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeGeofences()");
    }

    /**
     * Remove all geofences from local storage
     *
     * @param context Used to access local storage.
     */
    private static void removeOldGeofencesFromDatabase(Context context) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".removeOldGeofencesFromDatabase()");

        GeofenceDataSource accuratDatabase;
        try {
            accuratDatabase = new AccuratDatabase(DatabaseHelper.getInstance(context));
            final long numDeletedFences = accuratDatabase.removeGeofences();
            AccuratLogger.log(AccuratLogger.DATABASE, "Removed " + numDeletedFences + " geofences from database");
        } catch (Exception e) {
            AccuratLogger.log(AccuratLogger.ERROR, "Failed to remove geofences from database: " + e.getMessage());
        }
    }
    //</editor-fold>

    //<editor-fold desc="Helpers">

    /**
     * Get the geofences closest to a location from local storage
     *
     * @param context           context required to get local geofences
     * @param location          location to search geofences around for
     * @param box               box to search the geolocations in
     * @param geofenceIteration number of times this method has been called to get a result
     * @return the list of resulted geofences and search box used to find these geofences
     */
    private static GeofenceSelection getClosestGeofences(Context context, LocationInterface location, AccuratGeofenceRange box, int geofenceIteration) {
        List<AccuratGeofence> localFences = getGeofencesWithinBox(context, location, box);
        if (localFences.size() <= MAX_NEAR_GEOFENCES) {
            return new GeofenceSelection(location, localFences, box, geofenceIteration);
        }

        // Cut range in half to get less geofences
        Double latRange = box.getLatitudeRange() / 2;
        Double lngRange = box.getLongitudeRange() / 2;

        // Random values when ranges dip below minimum threshold
        // The proximity of geofences being so small means it doesn't really matter which one are selected
        if (latRange <= MIN_LAT_BOX_RANGE) {
            return new GeofenceSelection(location, localFences.subList(0, MAX_NEAR_GEOFENCES), box, geofenceIteration);
        }

        // Recursively search for a more appropriate set of geofences
        AccuratGeofenceRange newBox = new AccuratGeofenceRange(latRange, lngRange);
        return getClosestGeofences(context, location, newBox, geofenceIteration + 1);
    }

    private static List<AccuratGeofence> getGeofencesWithinBox(Context context, LocationInterface location, AccuratGeofenceRange box) {
        GeofenceDataSource dataSource = new AccuratDatabase(DatabaseHelper.getInstance(context));
        return dataSource.getGeofencesWithinBox(location, box);
    }

    /**
     * @return true when there are geofences in local storage
     */
    private static boolean hasGeofences(Context context) {
        GeofenceDataSource dataSource = new AccuratDatabase(DatabaseHelper.getInstance(context));
        return dataSource.getNumberOfGeofences() > 0;
    }

    /**
     * Helper method to indicate that the call has failed to complete.
     */
    private static void fail(@NonNull AccuratActionableCallback callback) {
        callback.onCompleted(false);
    }

    /**
     * Helper method used to find a geofence with its request ID
     *
     * @param context   Context used to access local storage
     * @param requestId Request ID of geofence to fetch from local storage
     * @return Geofence that has been found, or null in the case that it's not saved in the database
     */
    public static AccuratGeofence findGeofenceById(Context context, String requestId) {
        GeofenceDataSource dataSource = new AccuratDatabase(DatabaseHelper.getInstance(context));
        return dataSource.getGeofenceById(requestId);
    }

    /**
     * Notifies the manager that a geofence has been triggered, and acts upon it.
     *
     * @param context            Context required to access storage and show a notification
     * @param triggeringGeofence Geofence that has been triggered
     * @param isExit             Indicates whether the geofence was exited (true) or entered
     * @param isDwelling         Indicates whether the geofence is dwelled within, which caused the trigger
     */
    public static void triggerGeofence(Context context, Geofence triggeringGeofence, boolean isExit, boolean isDwelling) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".triggerGeofence()");
        if (TextUtils.isEmpty(triggeringGeofence.getRequestId())) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Request ID is empty, nothing to do");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".triggerGeofence()");

            return;
        }
        if (isExit) {
            if (META_GEOFENCE_ID.equals(triggeringGeofence.getRequestId())) {
                // Re-calculate geofences to monitor
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Exited meta geofence, recalculate geofences if there are more than " + MAX_NEAR_GEOFENCES);
            } else if (triggeringGeofence.getRequestId().startsWith(CURRENT_LOCATION_GEOFENCE_ID_PREFIX)) {
                // Redraw geofences & re-request locations
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Exited current location geofence, new location should be reported to AccuratLocationManager, nothing to do");
            } else {
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "Exited geofences, nothing to do");
            }
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".triggerGeofence()");

            return;
        }

        // Only retrieve & show notification when dwelling in a geofence
        if (!isDwelling) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Entered geofence with request ID " + triggeringGeofence.getRequestId() + ", nothing to do");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".triggerGeofence()");

            return;
        }

        // See whether the triggered geofence is in the database
        AccuratGeofence localFence = GeofencesManager.findGeofenceById(context, triggeringGeofence.getRequestId());
        if (localFence == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "Dwelling in unknown geofence (not found in database)");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".triggerGeofence()");

            return;
        }

        // Retrieve the notification for the triggered geofence
        AccuratLogger.log(AccuratLogger.GEOFENCE, "Dwelling in geofence with ID " + triggeringGeofence.getRequestId());
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Requesting a notification");
        GeofencesManager.showNotificationForGeofence(context, localFence, success -> {
            if (success) {
                AccuratLogger.log(AccuratLogger.NOTIFICATION, "Successfully posted notification for geofence with ID " + triggeringGeofence.getRequestId());
            } else {
                AccuratLogger.log(AccuratLogger.WARNING, "Failed to post notification for geofence with ID " + triggeringGeofence.getRequestId());
            }
        });
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".triggerGeofence()");
    }

    /**
     * Determines whether this geofence is the META-geofence surrounding all other geofences
     *
     * @param accuratGeofence Geofence to verify the identity of
     * @return true when this geofence is the META-geofence
     */
    public static boolean isMetaFence(AccuratGeofence accuratGeofence) {
        return accuratGeofence != null && META_GEOFENCE_ID.equals(accuratGeofence.getId());
    }

    /**
     * Determines whether this geofence is a META-geofence used to monitor the user's current location.
     *
     * @param accuratGeofence Geofence to verify the identity of
     * @return true when this geofence is a 'current location' META-geofence
     */
    public static boolean isCurrentLocationMetaFence(AccuratGeofence accuratGeofence) {
        return accuratGeofence != null && !TextUtils.isEmpty(accuratGeofence.getId())
                && accuratGeofence.getId().startsWith(CURRENT_LOCATION_GEOFENCE_ID_PREFIX);
    }

    /**
     * Determines whether the list of monitored geofences should be recalculated for a Geofence update.
     *
     * @param triggeredGeofences List of geofences in the {@link com.google.android.gms.location.GeofencingEvent}.
     * @return True when the monitored geofences should be updated, false if that is not the case.
     */
    public static boolean shouldUpdateMonitoredGeofences(List<Geofence> triggeredGeofences) {
        boolean containsNotificationGeofenceTrigger = false;
        for (Geofence triggeredGeofence : triggeredGeofences) {
            if (!triggeredGeofence.getRequestId().startsWith(CURRENT_LOCATION_GEOFENCE_ID_PREFIX)) {
                containsNotificationGeofenceTrigger = true;
                break;
            }
        }
        // Only meta geofences -> update monitored geofences since location should have changed
        if (!containsNotificationGeofenceTrigger) {
            return true;
        }

        // Verify whether the notification geofences in question were triggered less than 3 hours ago, in which case the monitored geofences may be ignored.
        String lastGeofenceTriggerId = mStorage.getString(StorageKeys.ACCURAT_GEOFENCES_LAST_TRIGGER, "");
        long lastGeofenceTriggerTimestamp = mStorage.getLong(StorageKeys.ACCURAT_GEOFENCES_LAST_TRIGGER_TIMESTAMP, 0L);
        // True when the same geofence is allowed to trigger a redraw of geofences
        boolean allowConsecutiveGeofenceTrigger = lastGeofenceTriggerTimestamp > System.currentTimeMillis() - GEOFENCE_TRIGGER_HIATUS;

        // Verify whether the list of geofences contain a geofence that should update the monitored geofences
        for (Geofence triggeredGeofence : triggeredGeofences) {
            boolean isMetaFence = triggeredGeofence.getRequestId().startsWith(CURRENT_LOCATION_GEOFENCE_ID_PREFIX);
            boolean isPreviousTrigger = triggeredGeofence.getRequestId().equals(lastGeofenceTriggerId);

            // Meta geofences cannot trigger a redraw
            if (isMetaFence) {
                continue;
            }

            if (!isPreviousTrigger) {
                // Different notification geofence, should trigger a redraw.
                mStorage.setValue(StorageKeys.ACCURAT_GEOFENCES_LAST_TRIGGER, triggeredGeofence.getRequestId())
                        .setValue(StorageKeys.ACCURAT_GEOFENCES_LAST_TRIGGER_TIMESTAMP, System.currentTimeMillis())
                        .commit();
                return true;
            } else if (allowConsecutiveGeofenceTrigger) {
                // Same notification geofence, but longer than three hours ago, should trigger a redraw.
                mStorage.setValue(StorageKeys.ACCURAT_GEOFENCES_LAST_TRIGGER_TIMESTAMP, System.currentTimeMillis())
                        .commit();
                return true;
            }
        }

        // List of geofences only had meta geofences and last triggered geofence, but was triggered less than 3 hours ago.
        return false;
    }

    private static void complete(AccuratCompletionCallback onComplete, boolean success) {
        if (onComplete != null) {
            onComplete.onCompleted(success);
        }
    }

    private static Date stringToDate(String dateString) {
        try {
            return GeofenceNotificationLog.DATE_FORMAT.parse(dateString);
        } catch (ParseException e) {
            return null;
        }
    }

    private static long getDayDifference(Date date1, String dateString) {
        Date date2 = stringToDate(dateString);
        if (date2 == null) {
            return Integer.MIN_VALUE;
        }

        long diffInMillis = date1.getTime() - date2.getTime();

        return diffInMillis / (1000 * 60 * 60 * 24);

    }
    //</editor-fold>

    // <editor-fold desc="Storage">
    // <editor-fold desc="Meta geofence storage">
    private static AccuratGeofence loadMetaGeofence(AccuratGeofence geofence) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".loadMetaGeofence()");
        // Only load when not already available
        if (geofence != null) {
            AccuratLogger.log(AccuratLogger.STORAGE, "No need to load meta geofence, already available");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadMetaGeofence()");

            return geofence;
        }

        if (mStorage == null) {
            // Storage is not initialised, can't load the meta geofence
            AccuratLogger.log(AccuratLogger.ERROR, "Can't load meta geofence, storage not initialised");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadMetaGeofence()");

            return null;
        }

        try {
            // Build the AccuratGeofence from the stored serialisation
            String json = mStorage.getString(StorageKeys.ACCURAT_GEOFENCES_META, null);
            AccuratLogger.log(AccuratLogger.STORAGE_DATA, "Loaded meta geofence: " + json);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadMetaGeofence()");

            return AccuratGeofence.fromJson(json);
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to load meta geofence: " + e.getMessage());
            e.printStackTrace();
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadMetaGeofence()");

            return null;
        }
    }

    private static void storeMetaGeofence(AccuratGeofence geofence) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".storeMetaGeofence()");
        if (mStorage == null) {
            // Storage is not initialised, can't store the meta geofence
            AccuratLogger.log(AccuratLogger.ERROR, "Can't store meta geofence, storage not initalised");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeMetaGeofence()");

            return;
        }

        if (geofence == null) {
            mStorage.setValue(StorageKeys.ACCURAT_GEOFENCES_META, null)
                    .commit();
            AccuratLogger.log(AccuratLogger.STORAGE_DATA, "Stored meta geofence: NULL");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeMetaGeofence()");

            return;
        }

        String serialisation = geofence.getJson().toString();
        mStorage.setValue(StorageKeys.ACCURAT_GEOFENCES_META, serialisation)
                .commit();
        AccuratLogger.log(AccuratLogger.STORAGE_DATA, "Stored meta geofence: " + serialisation);
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeMetaGeofence()");
    }
    // </editor-fold>

    // <editor-fold desc="Currently monitored geofences storage">
    private static List<AccuratGeofence> loadCurrentMonitoredGeofences(List<AccuratGeofence> geofences) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".loadCurrentMonitoredGeofences()");
        // Only load when not already available
        if (!CollectionUtils.isEmpty(geofences)) {
            AccuratLogger.log(AccuratLogger.STORAGE, "No need to load geofences, already available");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadCurrentMonitoredGeofences()");

            return geofences;
        }

        if (mStorage == null) {
            // Storage is not initialised, can't load the current monitored geofences
            AccuratLogger.log(AccuratLogger.ERROR, "Can't load geofences, storage not initialised");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadCurrentMonitoredGeofences()");

            return null;
        }

        try {
            // Build the List<AccuratGeofence> from the stored serialisation
            String json = mStorage.getString(StorageKeys.ACCURAT_GEOFENCES_CURRENT, null);

            if (json == null || json.isEmpty()) {
                AccuratLogger.log(AccuratLogger.STORAGE_DATA, "Loaded geofences: NULL");
                AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadCurrentMonitoredGeofences()");

                return null;
            }

            JSONArray jsonArray = new JSONArray(json);
            List<AccuratGeofence> currentFences = new ArrayList<>();
            for (int i = 0; i < jsonArray.length(); i++) {
                AccuratGeofence geofence = AccuratGeofence.fromJson(jsonArray.get(i).toString());
                if (geofence != null) {
                    currentFences.add(geofence);
                }
            }
            AccuratLogger.log(AccuratLogger.STORAGE_DATA, "Loaded " + currentFences + " geofences");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadCurrentMonitoredGeofences()");

            return currentFences;
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "Failed to load geofences: " + e.getMessage());
            e.printStackTrace();
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".loadCurrentMonitoredGeofences()");

            return null;
        }
    }

    private static void storeCurrentMonitoredGeofences(List<AccuratGeofence> geofences) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".storeCurrentMonitoredGeofences()");
        if (mStorage == null) {
            // Storage is not initialised, can't store the current monitored geofences
            AccuratLogger.log(AccuratLogger.ERROR, "Can't store monitored geofences, storage not initialised");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeCurrentMonitoredGeofences()");

            return;
        }

        if (geofences == null) {
            mStorage.setValue(StorageKeys.ACCURAT_GEOFENCES_CURRENT, null)
                    .commit();
            AccuratLogger.log(AccuratLogger.STORAGE_DATA, "Stored monitored geofences: NULL");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeCurrentMonitoredGeofences()");

            return;
        }

        JSONArray jsonArray = new JSONArray();
        for (AccuratGeofence geofence : geofences) {
            if (geofence != null) {
                jsonArray.put(geofence.getJson());
            }
        }

        String serialisation = jsonArray.toString();
        mStorage.setValue(StorageKeys.ACCURAT_GEOFENCES_CURRENT, serialisation)
                .commit();
        AccuratLogger.log(AccuratLogger.STORAGE_DATA, "Stored " + jsonArray.length() + " monitored geofences");
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".storeCurrentMonitoredGeofences()");
    }
    // </editor-fold>
    // </editor-fold>

    // <editor-fold desc="Hardcoded geofence">
    static void checkHardcodedGeofence(Context context, Location location) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".checkHardcodedGeofence()");
        if(location == null) {
            AccuratLogger.log(AccuratLogger.NONE, "Location is null, nothing to check");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".checkHardcodedGeofence()");

            return;
        }
        Location accuratOffices = new Location("HARDCODED");
        accuratOffices.setLatitude(51.01161);// Accurat
        accuratOffices.setLongitude(3.75125);// Accurat
//        accuratOffices.setLatitude(51.0158628);// Endare
//        accuratOffices.setLongitude(3.7346558);// Endare
        double maxDistance = 300;

        double distance = LocationUtils.distance(location.getLongitude(), location.getLatitude(), accuratOffices.getLongitude(), accuratOffices.getLatitude());
        AccuratLogger.log(AccuratLogger.NONE, "[" + location.getLatitude() + ", " + location.getLongitude() + "] is " + distance + " meters from Accurat offices [" + accuratOffices.getLatitude() + ", " + accuratOffices.getLongitude() +"]");
        if(distance <= maxDistance) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Close to the Accurat offices, going to trigger a notification");
            GeofenceNotification notification = new GeofenceNotification(-1, "{}", "Accurat offices", "This is a hardcoded geofence notification", "aqua", 0, null);
            showNotification(context, notification);
        } else {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Far from the Accurat offices, no need to show a notification");
        }

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".checkHardcodedGeofence()");
    }
    // </editor-fold>
}
