/*
 * Copyright (c) 2017 OpenLocate
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package ai.accurat.sdk.core;

import android.net.Uri;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import ai.accurat.sdk.config.Configuration;
import ai.accurat.sdk.constants.AccuratLocationKeys;
import ai.accurat.sdk.constants.ApiKeys;
import ai.accurat.sdk.constants.HttpMethod;
import ai.accurat.sdk.managers.AccuratConfigurationManager;

/**
 * @Both
 */
public final class LocationDispatcher {

    private static final String TAG = LocationDispatcher.class.getSimpleName();
    public static final String LOCATIONS_KEY = "locations";

    List<LocationInterface> postLocations(HttpClient httpClient, final OpenLocateBasedEndpoint endpoint, long sinceId, final LocationDataSource dataSource, final MultiProcessStorage storage) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".postLocations()");
        final List<LocationInterface> locations = dataSource.getSince(sinceId);
        if (locations == null || locations.isEmpty()) {
            AccuratLogger.log(AccuratLogger.DISPATCHER, "No locations to post");

            return null;
        }
        AccuratLogger.log(AccuratLogger.DISPATCHER, "Posting " + locations.size() + " locations");

        final String endpointUrl = endpoint.getUrl();
        final String body = getLocationsParam(locations).toString();
        AccuratLogger.log(AccuratLogger.NETWORK_REQUEST, "Calling " + endpointUrl);
        AccuratLogger.log(AccuratLogger.NETWORK_REQUEST_DATA, body);
        httpClient.post(
                endpointUrl,
                body,
                getEndpointHeaders(storage, endpoint, body),
                (request, response) -> {
                    AccuratLogger.logNetworkResponse(HttpMethod.POST, endpointUrl, response, false);
                    AccuratLogger.log(AccuratLogger.DISPATCHER, "Successfully posted " + locations.size() + " locations");
                }, (request, response) -> {
                    locations.clear();
                    AccuratLogger.logNetworkError(HttpMethod.POST, endpointUrl, response);
                    AccuratLogger.log(AccuratLogger.DISPATCHER, "Failed to post " + locations.size() + " locations");
                }
        );

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".postLocations()");
        return locations;
    }

    public static Map<String, String> getEndpointHeaders(MultiProcessStorage storage, OpenLocateBasedEndpoint endpoint, String body) {
        Map<String, String> headers = endpoint.getHeaders();
        if (headers == null) {
            headers = new HashMap<>();
        }
        if (!headers.containsKey(ApiKeys.Header.AUTHORIZATON)) {
            String path = Uri.parse(endpoint.getUrl()).getPath();
            headers.putAll(AccuratApi.getHeaders(storage, "POST", "application/json; charset=UTF-8",
                    AccuratApi.getEncodedRequestBody(body), path));
        }

        return headers;
    }

    private JSONObject getLocationsParam(List<LocationInterface> locationsToPost) {
        JSONObject jsonObject = new JSONObject();
        JSONArray jsonArray = new JSONArray();
        for (LocationInterface location : locationsToPost) {
            JSONObject json = location.getJson();
            try {
                json.put(AccuratLocationKeys.SDK_VERSION, Configuration.SDK_VERSION);
                json.put(AccuratLocationKeys.APP_VERSION, AccuratConfigurationManager.getAppVersion());
                json.put(AccuratLocationKeys.APP_VERSION_CODE, String.valueOf(AccuratConfigurationManager.getAppVersionCode()));
            } catch (JSONException e) {
                e.printStackTrace();
            }
            jsonArray.put(json);
        }

        try {
            jsonObject.put(LOCATIONS_KEY, jsonArray);
        } catch (JSONException e) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, TAG + ".getLocationsParam(): " + e.getMessage());
        }

        return jsonObject;
    }
}
