package ai.accurat.sdk.core;

import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.location.Location;
import android.os.Build;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.android.gms.location.FusedLocationProviderClient;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationServices;

import java.util.ArrayList;
import java.util.List;

import ai.accurat.sdk.callbacks.AccuratProcessCallback;

/**
 * Location tracking implementation
 */
public class LocationTrackerImpl implements LocationTrackerInterface {

    public static final String TAG = LocationTrackerImpl.class.getSimpleName();
    private LocationTrackerConfigurationInterface configuration;
    private LocationRequest locationRequest;
    private FusedLocationProviderClient fusedLocationProviderClient;

    LocationTrackerImpl() {
    }

    @Override
    public void initLocationTracking(Context context) {
        AccuratLogger.init(context);
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".initLocationTracking()");
        // Determine configuration
        if (!initConfiguration(context)) {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not initLocationTracking");
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".initLocationTracking()");
    }

    /**
     * Initializes the configuration
     *
     * @param context Context used to initialize the configuration settings
     * @return True when initialized, false if not
     */
    private boolean initConfiguration(Context context) {
        AccuratLogger.init(context);
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".initConfiguration()");
        AccuratSettings settings = AccuratSettingsManager.getSettings();
        if (settings == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "No settings available");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".initConfiguration(), returning false");
            return false;
        }
        createLocationRequest();

        List<OpenLocateBasedEndpoint> endpoints = settings.getEndpoints();
        if (endpoints == null || endpoints.isEmpty()) {
            AccuratLogger.log(AccuratLogger.WARNING, "No endpoints available");
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".initConfiguration(), returning false");
            return false;
        }

        this.configuration = new LocationTrackerConfigurationImpl.Builder(context, (ArrayList<OpenLocateBasedEndpoint>) endpoints).build();
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".initConfiguration(), returning true");

        return true;
    }

    @Override
    public void startLocationTracking(final Context context, final AccuratProcessCallback callback) {
        AccuratLogger.init(context);
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".startLocationTracking()");
        if (this.configuration == null && !initConfiguration(context)) {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not startLocationTracking, AccuratSettigns can't be read");
            callback.onProcessed(false);
            AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".startLocationTracking()");

            return;
        }

        // Request location updates
        try {
            this.fusedLocationProviderClient = LocationServices.getFusedLocationProviderClient(context);
            this.fusedLocationProviderClient.requestLocationUpdates(this.locationRequest, getPendingIntent(context));
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Requested location updates from FLP");
            callback.onProcessed(true);
        } catch (SecurityException e) {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not startLocationTracking, SecurityException: " + e.getMessage());
            callback.onProcessed(false);
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".startLocationTracking()");
    }

    @Override
    public void stopLocationTracking(Context context) {
        AccuratLogger.init(context);
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".stopLocationTracking()");

        if (this.fusedLocationProviderClient == null) {
            try {
                this.fusedLocationProviderClient = LocationServices.getFusedLocationProviderClient(context);
            } catch (SecurityException e) {
                AccuratLogger.log(AccuratLogger.ERROR, "Could not get FLP in stopLocationTracking, SecurityException: " + e.getMessage());
            }
        }
        if (this.fusedLocationProviderClient != null) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Removing location updates from FLP");
            this.fusedLocationProviderClient.removeLocationUpdates(getPendingIntent(context));
        } else {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not stopLocationTracking, because FLP is null");
        }

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".stopLocationTracking()");
    }

    @Override
    public LocationTrackerConfigurationInterface getConfiguration() {
        return this.configuration;
    }

    @Override
    public LocationInterface transformLocation(Context context, Location location, AdvertisingIdClient.Info advertisingInfo, UserActivity userActivity) {
        AccuratLogger.init(context);
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".transformLocation()");
        if (this.configuration == null && !initConfiguration(context)) {
            AccuratLogger.log(AccuratLogger.WARNING, "Could not initialise configuration");
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Returning default AccuratLocation");

            return new AccuratLocationBuilder()
                    .setLocation(location)
                    .setAdvertisingInfo(advertisingInfo)
                    .setInformationFields(InformationFieldsFactory.getInformationFields(Build.MANUFACTURER, Build.MODEL, "", Build.VERSION.SDK_INT >= Build.VERSION_CODES.M ? Build.VERSION.BASE_OS : "Android SDK " + Build.VERSION.SDK_INT, "", "", "", "", "", "", userActivity))
                    .build();
        }

        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".transformLocation()");
        return new AccuratLocationBuilder()
                .setLocation(location)
                .setAdvertisingInfo(advertisingInfo)
                .setInformationFields(InformationFieldsFactory.collectInformationFields(context, this.configuration, userActivity))
                .build();
    }

    /**
     * Stores parameters for requests to the FusedLocationProviderApi.
     */
    private void createLocationRequest() {
        this.locationRequest = new LocationRequest();

        this.locationRequest.setInterval(AccuratSettingsManager.getSettings().getLocationInterval() * 1000);// seconds => milliseconds

        // Sets the fastest rate for active location updates. This interval is exact, and your
        // application will never receive updates faster than this value.
        this.locationRequest.setFastestInterval(AccuratSettingsManager.getSettings().getFastestLocationInterval() * 1000);// seconds => milliseconds
        this.locationRequest.setPriority(LocationRequest.PRIORITY_HIGH_ACCURACY);

        // Sets the maximum time when batched location updates are delivered. Updates may be
        // delivered sooner than this interval.
        this.locationRequest.setMaxWaitTime(AccuratSettingsManager.getSettings().getMaxWaitTime());
        this.locationRequest.setSmallestDisplacement(AccuratSettingsManager.getSettings().getSmallestDisplacement());

        AccuratLogger.log(AccuratLogger.NONE, "locationRequest = " + locationRequest);
    }

    private PendingIntent getPendingIntent(Context context) {
        Intent intent = new Intent(context, LocationUpdatesBroadcastReceiver.class);
        intent.setAction(LocationUpdatesBroadcastReceiver.ACTION_LOCATION_UPDATES);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
            return PendingIntent.getBroadcast(context, 0, intent, PendingIntent.FLAG_UPDATE_CURRENT | PendingIntent.FLAG_MUTABLE);
        } else {
            return PendingIntent.getBroadcast(context, 0, intent, PendingIntent.FLAG_UPDATE_CURRENT);
        }
    }
}
