package ai.accurat.sdk.core;

import android.content.Context;
import androidx.annotation.NonNull;

import androidx.work.Worker;
import androidx.work.WorkerParameters;

import ai.accurat.sdk.Accurat;
import ai.accurat.sdk.data.RealmHelper;
import ai.accurat.sdk.data.models.Setting;
import ai.accurat.sdk.managers.RealmManager;

/**
 * @author Kenneth Saey
 * @since 02-04-2019 14:01
 */
public class TrackingRestartWorker extends Worker {

    public static final String TAG = TrackingRestartWorker.class.getSimpleName();

    public TrackingRestartWorker(@NonNull Context context, @NonNull WorkerParameters workerParams) {
        super(context, workerParams);
    }

    @NonNull
    @Override
    public Result doWork() {
        AccuratLogger.init(getApplicationContext());
        AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + " - doWork()");
        RealmManager.init(getApplicationContext());

        if (Accurat.isTrackingEnabled(getApplicationContext())) {
            AccuratLogger.log(AccuratLogger.NONE, "Tracking already enabled, nothing to do");
            logWorkDone();

            return Result.success();
        }

        AccuratLogger.log(AccuratLogger.WARNING, "Accurat is no longer tracking the location");

        MultiProcessStorage storage = MultiProcessStorage.getAccuratStorage(getApplicationContext());
        if (storage == null) {
            AccuratLogger.log(AccuratLogger.WARNING, "Storage not available, can't now if we should restart tracking");
            logWorkDone();

            return Result.failure();
        }

        boolean trackingWasStarted = RealmHelper.loadBooleanSetting(Setting.Keys.State.START_REQUESTED, true);
        AccuratLogger.log(AccuratLogger.NONE, "Realm trackingWasStarted = " + trackingWasStarted);
        if (!trackingWasStarted) {
            AccuratLogger.log(AccuratLogger.NONE, "Tracking was never started, so we shouldn't restart it");
            logWorkDone();

            return Result.success();
        }

        AccuratLogger.log(AccuratLogger.NONE, "Restarting location tracking...");
        try {
            Accurat.startAfterReboot(getApplicationContext());
        } catch (IllegalStateException e) {
            AccuratLogger.log(AccuratLogger.WARNING, "Accurat has not yet been initialised");
            AccuratLogger.log(AccuratLogger.NONE, "Initialising Accurat...");
            try {
                Accurat.initialize(getApplicationContext());
                AccuratLogger.log(AccuratLogger.NONE, "Restarting location tracking...");
                Accurat.startAfterReboot(getApplicationContext());
            } catch (IllegalStateException ise) {
                AccuratLogger.log(AccuratLogger.ERROR, ise.getMessage());
                AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + " - Work failed, returning Result.failure()");

                return Result.failure();
            }
        }
        AccuratLogger.log(AccuratLogger.NONE, "Tracking has been restarted");
        logWorkDone();

        return Result.success();
    }

    private void logWorkDone() {
        AccuratLogger.log(AccuratLogger.WORKMANAGER, TAG + " - Work done, returning Result.success()");
    }
}
