package ai.accurat.sdk.data;

import android.util.Log;

import androidx.annotation.Nullable;

import ai.accurat.sdk.core.AccuratLogger;
import ai.accurat.sdk.core.CampaignInteraction;
import ai.accurat.sdk.data.models.AccuratConfiguration;
import ai.accurat.sdk.data.models.ConsentManagerState;
import ai.accurat.sdk.data.models.ConsentModel;
import ai.accurat.sdk.data.models.DatabaseGeofence;
import ai.accurat.sdk.data.models.Setting;
import ai.accurat.sdk.data.models.TriggeredGeofence;
import ai.accurat.sdk.data.models.UserConsent;
import io.realm.DynamicRealm;
import io.realm.RealmMigration;
import io.realm.RealmObjectSchema;
import io.realm.RealmSchema;

public class AccuratMigration implements RealmMigration {

    private static final String TAG = AccuratMigration.class.getSimpleName();

    @Override
    public void migrate(DynamicRealm realm, long oldVersion, long newVersion) {
        AccuratLogger.log(AccuratLogger.METHOD_START, TAG + ".migrate(oldVersion = " + oldVersion + ", newVersion = " + newVersion + ")");
        RealmSchema schema = realm.getSchema();

        switch ((int) oldVersion) {
            case 0:
                migrate0To1(schema);
            case 1:
                migrate1To2(schema);
            case 2:
                migrate2To3(schema);
            case 3:
                migrate3To4(schema);
            case 4:
                migrate4To5(schema);
            case 5:
                migrate5To6(schema);
            case 6:
                migrate6To7(schema);
            case 7:
                migrate7To8(schema);
            case 8:
                migrate8to9(schema);
            case 9:
                migrate9to10(schema);
            case 10:
                migrate10to11(schema);
            case 11:
                migrate11to12(schema);
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, TAG + ".migrate()");
    }

    private void migrate0To1(RealmSchema schema) {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- migrate0To1()");
        // Create the Setting-table
        if (!schema.contains(Setting.RealmColumns._TABLE)) {
            schema.createWithPrimaryKeyField(Setting.RealmColumns._TABLE, Setting.RealmColumns.KEY, String.class)
                    .addField(Setting.RealmColumns.STRING_VALUE, String.class)
                    .addField(Setting.RealmColumns.BOOLEAN_VALUE, boolean.class)
                    .addField(Setting.RealmColumns.INTEGER_VALUE, int.class)
                    .addField(Setting.RealmColumns.LONG_VALUE, long.class);
        }
    }

    private void migrate1To2(RealmSchema schema) {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- migrate1To2()");
        // AccuratConfiguration table
        if (!schema.contains(AccuratConfiguration.RealmColumns._TABLE)) {
            AccuratLogger.log(AccuratLogger.NONE, "Creating AccuratConfiguration table (" + AccuratConfiguration.RealmColumns._TABLE + ")");
            schema.createWithPrimaryKeyField(AccuratConfiguration.RealmColumns._TABLE, AccuratConfiguration.RealmColumns.USERNAME, String.class)
                    .addField(AccuratConfiguration.RealmColumns.PASSWORD, String.class)
                    .addField(AccuratConfiguration.RealmColumns.NOTIFICATION_TARGET_PACKAGE, String.class)
                    .addField(AccuratConfiguration.RealmColumns.NOTIFICATION_TARGET_CLASS, String.class)
                    .addField(AccuratConfiguration.RealmColumns.GDPR_CONSENT, boolean.class)
                    .addField(AccuratConfiguration.RealmColumns.LOCATION_PERMISSION, boolean.class);
        } else {
            AccuratLogger.log(AccuratLogger.NONE, "AccuratConfiguration table already exists (" + AccuratConfiguration.RealmColumns._TABLE + ")");
        }
        RealmObjectSchema accuratConfigurationSchema = schema.get(AccuratConfiguration.RealmColumns._TABLE);
        AccuratLogger.log(AccuratLogger.NONE, "accuratConfigurationSchema = " + accuratConfigurationSchema);

        // ConsentModel table
        if (!schema.contains(ConsentModel.RealmColumns._TABLE)) {
            AccuratLogger.log(AccuratLogger.NONE, "Creating ConsentModel table (" + ConsentModel.RealmColumns._TABLE + ")");
            schema.createWithPrimaryKeyField(ConsentModel.RealmColumns._TABLE, ConsentModel.RealmColumns.ID, long.class)
                    .addField(ConsentModel.RealmColumns.TYPE_STRING, String.class)
//                    .setNullable(ConsentModel.RealmColumns.TYPE_STRING, false)
                    .addField(ConsentModel.RealmColumns.IS_SYSTEM_PERMISSION, boolean.class)
                    .addField(ConsentModel.RealmColumns.TITLE, String.class)
                    .setNullable(ConsentModel.RealmColumns.TITLE, false)
                    .addField(ConsentModel.RealmColumns.MESSAGE, String.class)
                    .setNullable(ConsentModel.RealmColumns.MESSAGE, false)
                    .addField(ConsentModel.RealmColumns.ACCEPT, String.class)
                    .setNullable(ConsentModel.RealmColumns.ACCEPT, false)
                    .addField(ConsentModel.RealmColumns.REFUTE, String.class)
                    .setNullable(ConsentModel.RealmColumns.REFUTE, false)
                    .addField(ConsentModel.RealmColumns.NEXT, String.class)
                    .setNullable(ConsentModel.RealmColumns.NEXT, false)
                    .addField(ConsentModel.RealmColumns.BRAND_COLOUR, String.class)
                    .setNullable(ConsentModel.RealmColumns.BRAND_COLOUR, false);
        } else {
            AccuratLogger.log(AccuratLogger.NONE, "ConsentModel table already exists (" + ConsentModel.RealmColumns._TABLE + ")");
        }
        RealmObjectSchema consentModelSchema = schema.get(ConsentModel.RealmColumns._TABLE);
        AccuratLogger.log(AccuratLogger.NONE, "consentModelSchema = " + consentModelSchema);

        // UserConsent table
        if (!schema.contains(UserConsent.RealmColumns._TABLE)) {
            AccuratLogger.log(AccuratLogger.NONE, "Creating UserConsent table (" + UserConsent.RealmColumns._TABLE + ")");
            schema.createWithPrimaryKeyField(UserConsent.RealmColumns._TABLE, UserConsent.RealmColumns.TYPE_STRING, String.class)
//                    .setNullable(UserConsent.RealmColumns.TYPE_STRING, false)
                    .addField(UserConsent.RealmColumns.ACCEPTED, boolean.class)
                    .addField(UserConsent.RealmColumns.REFUSE_COUNT, int.class)
                    .addField(UserConsent.RealmColumns.VERSION, long.class);
        } else {
            AccuratLogger.log(AccuratLogger.NONE, "UserConsent table already exists (" + UserConsent.RealmColumns._TABLE + ")");
        }
        RealmObjectSchema userConsentSchema = schema.get(UserConsent.RealmColumns._TABLE);
        AccuratLogger.log(AccuratLogger.NONE, "userConsentSchema = " + userConsentSchema);

        // ConsentManagerState table
        if (!schema.contains(ConsentManagerState.RealmColumns._TABLE)) {
            AccuratLogger.log(AccuratLogger.NONE, "Creating ConsentManagerState table (" + ConsentManagerState.RealmColumns._TABLE + ")");
            schema.createWithPrimaryKeyField(ConsentManagerState.RealmColumns._TABLE, ConsentManagerState.RealmColumns.ID, int.class)
                    .addField(ConsentManagerState.RealmColumns.FORCE_ASK_CONSENTS, boolean.class)
                    .addField(ConsentManagerState.RealmColumns.START_REQUESTED, boolean.class)
                    .addField(ConsentManagerState.RealmColumns.AD_ID, String.class)
                    .addField(ConsentManagerState.RealmColumns.AD_ID_TRACKING_ALLOWED, boolean.class)
                    .addRealmListField(ConsentManagerState.RealmColumns.CONSENT_MODELS, consentModelSchema)
                    .addRealmListField(ConsentManagerState.RealmColumns.USER_CONSENTS, userConsentSchema)
                    .addField(ConsentManagerState.RealmColumns.LAST_CONSENT_REQUEST_TIMESTAMP, long.class)
                    .addRealmObjectField(ConsentManagerState.RealmColumns.CONFIGURATION, accuratConfigurationSchema)
                    .addField(ConsentManagerState.RealmColumns.APP_CONSENTS_STRING, String.class)
//                    .setNullable(ConsentManagerState.RealmColumns.APP_CONSENTS_STRING, false)
                    .addField(ConsentManagerState.RealmColumns.CONSENT_FLOW_STATE_STRING, String.class)
//                    .setNullable(ConsentManagerState.RealmColumns.CONSENT_FLOW_STATE_STRING, false)
                    .addField(ConsentManagerState.RealmColumns.SHOULD_UPLOAD_USER_CONSENTS, boolean.class);
        } else {
            AccuratLogger.log(AccuratLogger.NONE, "ConsentManagerState table already exists (" + ConsentManagerState.RealmColumns._TABLE + ")");
        }
        RealmObjectSchema consentManagerStateSchema = schema.get(ConsentManagerState.RealmColumns._TABLE);
        AccuratLogger.log(AccuratLogger.NONE, "consentManagerStateSchema = " + consentManagerStateSchema);
    }

    private void migrate2To3(RealmSchema schema) {
        RealmObjectSchema accuratConfigurationSchema = schema.get(AccuratConfiguration.RealmColumns._TABLE);
        if (accuratConfigurationSchema == null) {
            return;
        }

        AccuratLogger.log(AccuratLogger.NONE, "Adding appVersion-column to AccuratConfiguration table");
        accuratConfigurationSchema.addField(AccuratConfiguration.RealmColumns.APP_VERSION, String.class);
    }

    private void migrate3To4(RealmSchema schema) {
        RealmObjectSchema campaignInteractionSchema = schema.get(CampaignInteraction.RealmKeys._TABLE);
        if (campaignInteractionSchema == null) {
            return;
        }

        AccuratLogger.log(AccuratLogger.NONE, "Adding id, start & end column to the CampaignInteraction table");
        if (!campaignInteractionSchema.hasField(CampaignInteraction.RealmKeys.ID)) {
            campaignInteractionSchema.addField(CampaignInteraction.RealmKeys.ID, String.class);
        }
        if (!campaignInteractionSchema.hasField(CampaignInteraction.RealmKeys.START)) {
            campaignInteractionSchema.addField(CampaignInteraction.RealmKeys.START, String.class);
        }
        if (!campaignInteractionSchema.hasField(CampaignInteraction.RealmKeys.END)) {
            campaignInteractionSchema.addField(CampaignInteraction.RealmKeys.END, String.class);
        }
    }

    private void migrate4To5(RealmSchema schema) {
        RealmObjectSchema accuratConfigurationSchema = schema.get(AccuratConfiguration.RealmColumns._TABLE);
        if (accuratConfigurationSchema == null) {
            return;
        }

        AccuratLogger.log(AccuratLogger.NONE, "Adding hasDebugLogsEnabled-column to AccuratConfiguration table");
        if (!accuratConfigurationSchema.hasField(AccuratConfiguration.RealmColumns.HAS_DEBUG_LOGS_ENABLED)) {
            accuratConfigurationSchema.addField(AccuratConfiguration.RealmColumns.HAS_DEBUG_LOGS_ENABLED, boolean.class);
        }
    }

    private void migrate5To6(RealmSchema schema) {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- migrate5To6()");
        if (!schema.contains(DatabaseGeofence.RealmColumns._TABLE)) {
            schema.createWithPrimaryKeyField(DatabaseGeofence.RealmColumns._TABLE, DatabaseGeofence.RealmColumns.ID, String.class)
                    .setNullable(DatabaseGeofence.RealmColumns.ID, false)
                    .addField(DatabaseGeofence.RealmColumns.LONGITUDE, Double.class)
                    .setNullable(DatabaseGeofence.RealmColumns.LONGITUDE, false)
                    .addField(DatabaseGeofence.RealmColumns.LATITUDE, Double.class)
                    .setNullable(DatabaseGeofence.RealmColumns.LATITUDE, false)
                    .addField(DatabaseGeofence.RealmColumns.RADIUS, Integer.class)
                    .setNullable(DatabaseGeofence.RealmColumns.RADIUS, false);
        }
    }

    private void migrate6To7(RealmSchema schema) {
        RealmObjectSchema databaseGeofenceSchema = schema.get(DatabaseGeofence.RealmColumns._TABLE);
        if (databaseGeofenceSchema == null) {
            return;
        }

        databaseGeofenceSchema.addField(DatabaseGeofence.RealmColumns.ENTERED_AT, Long.class)
                .setNullable(DatabaseGeofence.RealmColumns.ENTERED_AT, false);
    }

    private void migrate7To8(RealmSchema schema) {
        RealmObjectSchema accuratConfigurationSchema = schema.get(AccuratConfiguration.RealmColumns._TABLE);
        if (accuratConfigurationSchema == null) {
            return;
        }

        AccuratLogger.log(AccuratLogger.NONE, "Adding appVersionCode-column to AccuratConfiguration table");
        accuratConfigurationSchema.addField(AccuratConfiguration.RealmColumns.APP_VERSION_CODE, int.class);
    }

    private void migrate8to9(RealmSchema schema) {
        RealmObjectSchema consentModelSchema = schema.get(ConsentModel.RealmColumns._TABLE);
        if (consentModelSchema == null) {
            return;
        }

        AccuratLogger.log(AccuratLogger.NONE, "Adding maxRefuseCount-column to ConsentModel table");
        if (!consentModelSchema.hasField(ConsentModel.RealmColumns.MAX_REFUSE_COUNT)) {
            consentModelSchema.addField(ConsentModel.RealmColumns.MAX_REFUSE_COUNT, int.class);
        }
    }

    private void migrate9to10(RealmSchema schema) {
        RealmObjectSchema accuratConfigurationSchema = schema.get(AccuratConfiguration.RealmColumns._TABLE);
        if (accuratConfigurationSchema == null) {
            return;
        }

        AccuratLogger.log(AccuratLogger.NONE, "Adding featureSwitches-column to AccuratConfiguration table");
        accuratConfigurationSchema.addRealmDictionaryField(AccuratConfiguration.RealmColumns.FEATURE_SWITCHES, boolean.class);
    }

    private void migrate10to11(RealmSchema schema) {
        // TriggeredGeofence table
        if (!schema.contains(TriggeredGeofence.RealmColumns.TABLE)) {
            AccuratLogger.log(AccuratLogger.NONE, "Creating TriggeredGeofence table (" + TriggeredGeofence.RealmColumns.TABLE + ")");
            schema.createWithPrimaryKeyField(TriggeredGeofence.RealmColumns.TABLE, TriggeredGeofence.RealmColumns.ID, String.class)
                    .addField(TriggeredGeofence.RealmColumns.TRIGGER_TYPE, String.class)
                    .addField(TriggeredGeofence.RealmColumns.TIMESTAMP, String.class)
                    .addField(TriggeredGeofence.RealmColumns.DATA, String.class);
        } else {
            AccuratLogger.log(AccuratLogger.NONE, "TriggeredGeofence table already exists (" + TriggeredGeofence.RealmColumns.TABLE + ")");
        }
    }

    private void migrate11to12(RealmSchema schema) {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- migrate11to12()");

        RealmObjectSchema accuratConfigurationSchema = schema.get(AccuratConfiguration.RealmColumns._TABLE);
        changeNullabilityTo(
                accuratConfigurationSchema,
                AccuratConfiguration.RealmColumns.FEATURE_SWITCHES,
                true
        );

        RealmObjectSchema consentManagerStateSchema = schema.get(ConsentManagerState.RealmColumns._TABLE);
        changeNullabilityTo(
                consentManagerStateSchema,
                ConsentManagerState.RealmColumns.APP_CONSENTS_STRING,
                false
        );
        changeNullabilityTo(
                consentManagerStateSchema,
                ConsentManagerState.RealmColumns.CONSENT_FLOW_STATE_STRING,
                false
        );

        RealmObjectSchema consentModelSchema = schema.get(ConsentModel.RealmColumns._TABLE);
        changeNullabilityTo(
                consentModelSchema,
                ConsentModel.RealmColumns.TYPE_STRING,
                false
        );

        RealmObjectSchema triggeredGeofenceSchema = schema.get(TriggeredGeofence.RealmColumns.TABLE);
        changeNullabilityTo(
                triggeredGeofenceSchema,
                TriggeredGeofence.RealmColumns.ID,
                false
        );
        changeNullabilityTo(
                triggeredGeofenceSchema,
                TriggeredGeofence.RealmColumns.TRIGGER_TYPE,
                false
        );
        changeNullabilityTo(
                triggeredGeofenceSchema,
                TriggeredGeofence.RealmColumns.TIMESTAMP,
                false
        );
        changeNullabilityTo(
                triggeredGeofenceSchema,
                TriggeredGeofence.RealmColumns.DATA,
                false
        );

        RealmObjectSchema userConsentSchema = schema.get(UserConsent.RealmColumns._TABLE);
        changeNullabilityTo(
                userConsentSchema,
                UserConsent.RealmColumns.TYPE_STRING,
                false
        );
    }

    private void changeNullabilityTo(RealmObjectSchema schema, String fieldName, boolean isNullable) {
        if (schema == null || !schema.hasField(fieldName)) {
            return;
        }

        if (schema.isNullable(fieldName) == isNullable) {
            return;
        }

        try {
            schema.setNullable(fieldName, isNullable);
        } catch (Exception e) {
            String message = e.getMessage();
            Log.e(TAG, "Failed to change " + fieldName + " nullability to " + isNullable + ":" + ((message == null) ? "" : message));
        }
    }

    @Override
    public int hashCode() {
        return 37;
    }

    @Override
    public boolean equals(@Nullable Object obj) {
        return (obj instanceof AccuratMigration);
    }
}
