package ai.accurat.sdk.managers

import ai.accurat.sdk.R
import ai.accurat.sdk.callbacks.AccuratProcessCallback
import ai.accurat.sdk.config.Configuration
import ai.accurat.sdk.constants.*
import ai.accurat.sdk.core.*
import ai.accurat.sdk.data.enums.ConsentFlowState
import ai.accurat.sdk.data.enums.ConsentFlowState.*
import ai.accurat.sdk.data.enums.ConsentType
import ai.accurat.sdk.data.exceptions.UninitialisedException
import ai.accurat.sdk.data.models.*
import ai.accurat.sdk.data.save
import ai.accurat.sdk.viewholders.ConsentDialogViewHolder
import android.Manifest
import android.annotation.SuppressLint
import android.app.Activity
import android.app.AlertDialog
import android.content.Context
import android.content.pm.PackageManager
import android.os.Build
import android.view.View
import androidx.core.app.ActivityCompat
import com.android.volley.RequestQueue
import com.android.volley.Response
import com.android.volley.toolbox.JsonObjectRequest
import com.android.volley.toolbox.Volley
import com.google.android.gms.ads.identifier.AdvertisingIdClient
import io.realm.RealmList
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject
import java.lang.reflect.Method
import java.util.*
import kotlin.coroutines.Continuation
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

@SuppressLint("StaticFieldLeak")
object AccuratManager {

    private val TAG: String = AccuratManager::class.java.simpleName
    private const val ANDROID_BELOW_10_LOCATION_PERMISSION_REQUESTED: Int =
        RequestCodes.BASE_PERMISSION + 1
    private const val ANDROID_10_LOCATION_PERMISSION_REQUESTED: Int =
        RequestCodes.BASE_PERMISSION + 2
    private const val ANDROID_ABOVE_10_FOREGROUND_LOCATION_PERMISSION_REQUESTED: Int =
        RequestCodes.BASE_PERMISSION + 3
    private const val ANDROID_ABOVE_10_BACKGROUND_LOCATION_PERMISSION_REQUESTED: Int =
        RequestCodes.BASE_PERMISSION + 4

    var state: ConsentManagerState = ConsentManagerState.load()
    lateinit var context: Context
    lateinit var activity: Activity
    private lateinit var storage: MultiProcessStorage
    private var requestQueue: RequestQueue? = null
    private var startedCallback: AccuratProcessCallback? = null
    private var consentsAskedCallback: Runnable? = null

    // <editor-fold desc="Initialisation">
    fun init(context: Context) {
        if (isInitialised()) {
            return
        }
        this.context = context
        AccuratLogger.init(context)
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.init()")
        AccuratLogger.log(AccuratLogger.NONE, "Android version = ${androidVersion()}")
        storage = MultiProcessStorage.getStorage(context, StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE)
        requestQueue = Volley.newRequestQueue(context)
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.init()")
    }
    // </editor-fold>

    // <editor-fold desc="Step 0 - Preparation">
    fun forceAskConsents(force: Boolean = true) {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 0 - Preparation")
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.forceAskConsents($force)")
        load()
        state.forceAskConsents = force
        save()
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.forceAskConsents($force)")
    }

    @Throws(UninitialisedException::class)
    fun startTracking(activity: Activity, startedCallback: AccuratProcessCallback? = null) {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 0 - Preparation")
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.startTracking()")
        if (!isInitialised()) {
            throw UninitialisedException("context")
        }
        load()
        this.activity = activity

        this.consentsAskedCallback = null
        this.startedCallback = startedCallback
        state.startRequested = true
        state isAt STARTED
        save()

        checkSettings(activity)
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.startTracking()")
    }

    fun stopTracking() {
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.stopTracking()")
        if (!isInitialised()) {
            throw UninitialisedException("context")
        }
        load()
        state.startRequested = false
        save()
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.stopTracking()")
    }

    fun canStartFromBackground(context: Context): Boolean {
        init(context)

        return state.adId != null && state.adIdTrackingAllowed && state.isAccepted(ConsentType.GDPR) && hasBackgroundPermission()
    }

    fun askConsents(onComplete: Runnable) {
        consentsAskedCallback = onComplete
        checkAdId()
    }

    private fun checkSettings(context: Context) {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 0 - Fetch settings")
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.checkSettings()")
        AccuratSettingsManager.init(context)
        AccuratSettingsManager.fetchSettings { success ->
            if (!success) {
                AccuratLogger.log(AccuratLogger.WARNING, "Failed to fetch settings, continuing without")
                checkAdId()
                AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.checkSettings()")

                return@fetchSettings
            }

            val settings = AccuratSettingsManager.getSettings()
            if (settings != null && !settings.isSdkEnabled) {
                AccuratLogger.log(AccuratLogger.WARNING, "SDK is disabled in settings, finishing AccuratManager")
                stopSdk(context)
                finish(false)
                AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.checkSettings()")

                return@fetchSettings
            }

            AccuratLogger.log(AccuratLogger.SDK_FLOW, "SDK is enabled in settings, continuing")
            AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.checkSettings()")
            checkAdId()
        }
    }

    private fun stopSdk(context: Context) {
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.stopSdk()")
        AccuratLocationManager
            .getInstance()
            .apply {
                init(context)
            }
            .stop(context)
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.stopSdk()")
    }
    // </editor-fold>

    // <editor-fold desc="Step 1 - Check Ad ID">
    fun checkAdId() {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 1 - Check Ad ID")
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.checkAdId()")
        state isAt CHECKING_AD_ID
        CoroutineScope(Dispatchers.Default).launch {
            try {
                val adId = loadAdId()
                withContext(Dispatchers.Main) {
                    if (adId == null || state.adId != adId.id || state.adIdTrackingAllowed != !adId.isLimitAdTrackingEnabled) {
                        AccuratLogger.log(
                            AccuratLogger.WARNING, "Clearing server consents because: ${
                                when {
                                    adId == null -> "adId == null"
                                    state.adId != adId.id -> "adId changed from ${state.adId} to ${adId.id}"
                                    else -> "tracking allowed changed from ${state.adIdTrackingAllowed} to ${!adId.isLimitAdTrackingEnabled}"
                                }
                            }"
                        )
                        clearServerConsents()
                    }

                    if (adId != null) {
                        state.adId = adId.id
                        state.adIdTrackingAllowed = !adId.isLimitAdTrackingEnabled
                    }
                    save()
                    if (state.adIdTrackingAllowed) {
                        uploadUser()
                    }

                    fetchConsentModels()
                    AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.checkAdId()")
                }
            } catch (e: Exception) {
                withContext(Dispatchers.Main) {
                    AccuratLogger.log(AccuratLogger.ERROR, "Could not load ad ID: ${e.message}")
                    finish(false)
                }
            }
        }
    }

    private fun clearServerConsents() {
        /**
         * This is a no-op, because:
         * - If the Ad ID changed, the new Ad ID will be uploaded and used
         * - If only limit tracking was enabled, we shouldn't communicate about the Ad ID any more
         *   and Accurat will be stopped at the beginning of Step 2 - Fetch consent models
         */
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.clearSeverConsents()")
        AccuratLogger.log(
            AccuratLogger.NONE, "This is a no-op, because:[br]" +
                    "- If the Ad ID changed, the new Ad ID will be uploaded and used[br]" +
                    "- If only limit tracking was enabled, we shouldn't communicate about the Ad ID any more and Accurat will be stopped at the beginning of Step 2 - Fetch consent models"
        )
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.clearSeverConsents()")
    }

    private fun loadAdId(): AdvertisingIdClient.Info? =
        AdvertisingIdClient.getAdvertisingIdInfo(context)

    private fun uploadUser() {
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.uploadUser()")
        AccuratUserManager.init(context)
        if (hasUserChanged()) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "User has changed, uploading...")
            AccuratUserManager.upload()
            storage.setValue(
                StorageKeys.CACHED_USER,
                AccuratUserManager.getUser().toJson().toString()
            )
                .commit()
        } else {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "User has not changed, no need to upload")
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.uploadUser()")
    }

    private fun hasUserChanged(): Boolean {
        val storedUser = storage.getString(StorageKeys.CACHED_USER, "")
        val user = AccuratUserManager.getUser().toJson().toString()
        AccuratLogger.log(AccuratLogger.NONE, "storedUser = $storedUser")
        AccuratLogger.log(AccuratLogger.NONE, "user = $user")

        return storedUser != user
    }
    // </editor-fold>

    // <editor-fold desc="Step 2 - Fetch consent models">
    private fun fetchConsentModels() {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 2 - Fetch consent models")
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.fetchConsentModels()")
        state isAt FETCHING_CONSENT_MODELS

        if (!state.adIdTrackingAllowed) {
            AccuratLogger.log(AccuratLogger.ERROR, "Ad ID tracking not allowed")
            finish(false)

            return
        }

        CoroutineScope(Dispatchers.IO).launch {
            val serverConsentModels: List<ConsentModel>? = downloadConsentModels()
            AccuratLogger.log(AccuratLogger.NONE, "serverConsentModels = $serverConsentModels")
            withContext(Dispatchers.Main) {
                if (serverConsentModels == null) {
                    AccuratLogger.log(AccuratLogger.ERROR, "serverConsentModels are null")
                    finish(false)

                    return@withContext
                }
                state.setConsentModels(serverConsentModels)
                save()

                fetchUserConsents()
            }
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.fetchConsentModels()")
    }

    // <editor-fold desc="API - Consent Models">
    private suspend fun downloadConsentModels() =
        suspendCoroutine<List<ConsentModel>?> { continuation ->
            AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.downloadConsentModels()")
            if (!AccuratApi.isNetworkAvailable(context)) {
                AccuratLogger.log(
                    AccuratLogger.NETWORK,
                    "Network not available, cancelling downloadConsentModels()"
                )
                continuation.resume(null)

                return@suspendCoroutine
            }

            val urlParameters =
                hashMapOf<String, Any>(ApiKeys.Url.LANGUAGE_KEY to AccuratUserManager.getUserLanguage())
            val request = getConsentModelsRequest(
                continuation,
                urlParameters
            )

            AccuratLogger.logNetworkRequest(
                HttpMethod.GET,
                AccuratEndpoints.GET_CONSENTS.getUrl(urlParameters)
            )
            request.prepare(TAG)
                .enqueue()
            AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.downloadConsentModels()")
        }

    private fun getConsentModelsRequest(
        continuation: Continuation<List<ConsentModel>?>,
        urlParameters: HashMap<String, Any>,
    ): JsonObjectRequest = object : JsonObjectRequest(
        Method.GET,
        AccuratEndpoints.GET_CONSENTS.getUrl(urlParameters),
        null,
        onConsentModelsResponse(continuation),
        onConsentModelsError(continuation)
    ) {
        override fun getHeaders(): MutableMap<String, String> = AccuratApi.getHeaders(
            storage,
            "GET",
            "",
            AccuratApi.getEncodedRequestBody(""),
            AccuratEndpoints.GET_CONSENTS.getPath(urlParameters)
        )
    }

    private fun onConsentModelsResponse(continuation: Continuation<List<ConsentModel>?>): Response.Listener<JSONObject> =
        Response.Listener { response ->
            AccuratLogger.logNetworkResponse(
                HttpMethod.GET,
                Configuration.ENDPOINT_GET_CONSENTS,
                response,
                false
            )
            if (response == null || !response.has(ApiKeys.GetConsents.Response.DATA)) {
                AccuratLogger.log(
                    AccuratLogger.ERROR,
                    "No ${ApiKeys.GetConsents.Response.DATA}-field"
                )
                continuation.resume(null)

                return@Listener
            }

            try {
                val data = response.getJSONArray(ApiKeys.GetConsents.Response.DATA)
                val models: ArrayList<ConsentModel> = arrayListOf()

                (0 until data.length()).forEach { index ->
                    models.add(ConsentModel.fromServerJson(data.getJSONObject(index)))
                }

                continuation.resume(models)
            } catch (e: JSONException) {
                AccuratLogger.log(
                    AccuratLogger.JSON_ERROR,
                    "Failed to parse consent models: ${e.message}"
                )
                continuation.resume(null)
            } catch (e: Exception) {
                AccuratLogger.log(
                    AccuratLogger.ERROR,
                    "Unexpected error parsing consents models: ${e.message}"
                )
                continuation.resume(null)
            }
        }

    private fun onConsentModelsError(continuation: Continuation<List<ConsentModel>?>): Response.ErrorListener =
        Response.ErrorListener { error ->
            AccuratLogger.logNetworkError(
                HttpMethod.GET,
                Configuration.ENDPOINT_GET_CONSENTS,
                error
            )
            continuation.resume(null)
        }
    // </editor-fold>
    // </editor-fold>

    // <editor-fold desc="Step 3 - Fetch user consents">
    private fun fetchUserConsents() {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 3 - Fetch user consents")
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.fetchUserConsents()")
        state isAt FETCHING_USER_CONSENTS
        if (!state.adIdTrackingAllowed) {
            AccuratLogger.log(AccuratLogger.ERROR, "Ad ID tracking not allowed")
            finish(false)

            return
        }

        if (state.shouldUploadUserConsents) {
            AccuratLogger.log(
                AccuratLogger.SDK_FLOW,
                "Failed to upload user consents previously, trying now instead of fetching them from the server"
            )
            sendConsentsToServer(true)
            AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.fetchUserConsents()")
            return
        }
        CoroutineScope(Dispatchers.IO).launch {
            val userConsents: List<UserConsent>? = downloadUserConsents()
            AccuratLogger.log(AccuratLogger.NONE, "userConsents = $userConsents")
            withContext(Dispatchers.Main) {
                userConsents?.let {
                    state.setUserConsents(userConsents)
                    save()
                }

                checkConsents()
            }
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.fetchUserConsents()")
    }

    // <editor-fold desc="API - Fetch user consents">
    private suspend fun downloadUserConsents() =
        suspendCoroutine<List<UserConsent>?> { continuation ->
            AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.downloadUserConsents()")
            if (!AccuratApi.isNetworkAvailable(context)) {
                AccuratLogger.log(
                    AccuratLogger.NETWORK,
                    "Network not available, cancelling downloadUserConsents()"
                )
                continuation.resume(null)

                return@suspendCoroutine
            }

            if (state.adId == null) {
                finish(false)

                return@suspendCoroutine
            }
            val adId = state.adId!!
            val urlParameters = hashMapOf<String, Any>(ApiKeys.Url.AD_ID to adId)
            val request = getUserConsentsRequest(
                continuation,
                urlParameters
            )

            AccuratLogger.logNetworkRequest(
                HttpMethod.GET,
                AccuratEndpoints.GET_CONSENT.getUrl(urlParameters)
            )
            request.prepare(TAG)
                .enqueue()
            AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.downloadUserConsents()")
        }

    private fun getUserConsentsRequest(
        continuation: Continuation<List<UserConsent>?>,
        urlParameters: java.util.HashMap<String, Any>,
    ): JsonObjectRequest = object : JsonObjectRequest(
        Method.GET,
        AccuratEndpoints.GET_CONSENT.getUrl(urlParameters),
        null,
        onUserConsentsResponse(continuation),
        onUserConsentsError(continuation)
    ) {
        override fun getHeaders(): MutableMap<String, String> = AccuratApi.getHeaders(
            storage,
            "GET",
            "",
            AccuratApi.getEncodedRequestBody(""),
            AccuratEndpoints.GET_CONSENT.getPath(urlParameters)
        )
    }

    private fun onUserConsentsResponse(continuation: Continuation<List<UserConsent>?>): Response.Listener<JSONObject> =
        Response.Listener { response ->
            AccuratLogger.logNetworkResponse(
                HttpMethod.GET,
                Configuration.ENDPOINT_GET_CONSENT,
                response,
                false
            )
            if (response == null || !response.has(ApiKeys.GetConsent.Response.DATA)) {
                AccuratLogger.log(
                    AccuratLogger.ERROR,
                    "No ${ApiKeys.GetConsent.Response.DATA}-field"
                )
                continuation.resume(null)

                return@Listener
            }

            try {
                val data = response.getJSONArray(ApiKeys.GetConsent.Response.DATA)
                val consents: ArrayList<UserConsent> = arrayListOf()

                (0 until data.length()).forEach { index ->
                    consents.add(UserConsent.fromServerJson(data.getJSONObject(index)))
                }

                continuation.resume(consents)
            } catch (e: JSONException) {
                AccuratLogger.log(
                    AccuratLogger.JSON_ERROR,
                    "Failed to parse user consents: ${e.message}"
                )
                continuation.resume(null)
            } catch (e: Exception) {
                AccuratLogger.log(
                    AccuratLogger.ERROR,
                    "Unexpected error parsing user consents: ${e.message}"
                )
                continuation.resume(null)
            }
        }

    private fun onUserConsentsError(continuation: Continuation<List<UserConsent>?>): Response.ErrorListener =
        Response.ErrorListener { error ->
            AccuratLogger.logNetworkError(HttpMethod.GET, Configuration.ENDPOINT_GET_CONSENT, error)
            continuation.resume(null)
        }
    // </editor-fold>
    // </editor-fold>

    // <editor-fold desc="Step 4 - Check consents">
    private fun checkConsents() {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 4 - Checking consents")
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.checkConsents()")
        state isAt CHECKING_CONSENTS
        if (shouldAskConsents()) {
            checkGdpr()
        } else {
            autoCheckPermissions()
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.checkConsents()")
    }

    // <editor-fold desc="Step 4a - Auto-check permissions">
    private fun autoCheckPermissions() {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 4a - Auto-check permissions")
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.autoCheckPermissions()")
        if (isAndroidBelow10()) {
            // Android < 10
            AccuratLogger.log(AccuratLogger.NONE, "Android < 10")
            if (hasPrecisePermission()) {
                state.accept(ConsentType.PRECISE_PERMISSION)
            } else {
                state.autoRefuse(ConsentType.PRECISE_PERMISSION)
            }
            if (hasForegroundPermission()) {
                state.accept(ConsentType.ALWAYS_LOCATION, ConsentType.IN_APP_LOCATION)
            } else {
                state.autoRefuse(ConsentType.ALWAYS_LOCATION, ConsentType.IN_APP_LOCATION)
            }
        } else {
            // Android >= 10
            AccuratLogger.log(AccuratLogger.NONE, "Android >= 10")
            if (hasPrecisePermission()) {
                state.accept(ConsentType.PRECISE_PERMISSION)
            } else {
                state.autoRefuse(ConsentType.PRECISE_PERMISSION)
            }
            if (hasForegroundPermission()) {
                state.accept(ConsentType.IN_APP_LOCATION)
            } else {
                state.autoRefuse(ConsentType.IN_APP_LOCATION)
            }
            if (hasBackgroundPermission()) {
                state.accept(ConsentType.ALWAYS_LOCATION)
            } else {
                state.autoRefuse(ConsentType.ALWAYS_LOCATION)
            }
        }
        save()
        sendConsentsToServer()
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.autoCheckPermissions()")
    }
    // </editor-fold>

    // <editor-fold desc="Step 4b - GDPR consent">
    private fun checkGdpr() {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 4b - GDPR consent")
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.checkGdpr()")
        if (state.isAccepted(ConsentType.GDPR)) {
            sendGeneralInfoToServer()
            checkAndroidBelow10LocationPermissions()
        } else {
            runGdprConsent()
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.checkGdpr()")
    }

    private fun runGdprConsent() {
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.runGdprConsent()")
        if (!state.mayAsk(ConsentType.GDPR)) {
            autoCheckPermissions()

            return
        }

        if (state.configuration?.usesGdprConsentFeature() != false) {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "SDK uses GDPR consent feature")
            askGdprConsent()

            return
        }

        AccuratLogger.log(AccuratLogger.SDK_FLOW, "SDK doesn't use GDPR consent feature")
        if (state.hasAppConsent(ConsentType.GDPR) || state.isAccepted(ConsentType.GDPR)) {
            state.accept(ConsentType.GDPR)
            save()
            sendGeneralInfoToServer()
            checkAndroidBelow10LocationPermissions()

            return
        }

        state.autoRefuse(ConsentType.GDPR)
        save()
        autoCheckPermissions()
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.runGdprConsent()")
    }

    private fun askGdprConsent() {
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.askGdprConsent()")
        val viewHolder =
            ConsentDialogViewHolder(View.inflate(context, R.layout.dialog_consent, null))
        viewHolder.setConsentModel(state.consentModels.find(ConsentType.GDPR))

        val consentDialog = AlertDialog.Builder(context)
            .setView(viewHolder.rootView)
            .setCancelable(false)
            .setOnCancelListener {
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "GdprConsentDialog.onCancel()")
                state.refuse(ConsentType.GDPR)
                setAndPushConsentsToServer(ConsentType.GDPR, false)
                autoCheckPermissions()
            }
            .create()
        consentDialog.setCanceledOnTouchOutside(false)

        viewHolder.setPositiveButtonHandler {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "GdprConsentDialog.onPositive()")
            consentDialog.dismiss()
            state.accept(ConsentType.GDPR)
            save()
            setAndPushConsentsToServer(ConsentType.GDPR, true) {
                sendGeneralInfoToServer()
            }
            checkAndroidBelow10LocationPermissions()
        }.setNegativeButtonHandler {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "GdprConsentDialog.onNegative()")
            consentDialog.dismiss()
            state.refuse(ConsentType.GDPR)
            save()
            setAndPushConsentsToServer(ConsentType.GDPR, false)
            autoCheckPermissions()
        }

        ConsentManagerState.updateConsentRequestTimestamp()
        load()
        try {
            if (!activity.isFinishing) {
                consentDialog.show()
                viewHolder.enableDialogLinks()
            } else {
                AccuratLogger.log(
                    AccuratLogger.WARNING,
                    "Could not show GdprConsentDialog: Activity is finishing"
                )
            }
        } catch (e: Exception) {
            AccuratLogger.log(AccuratLogger.ERROR, "Could not show GdprConsentDialog: ${e.message}")
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.askGdprConsent()")
    }
    // </editor-fold>

    // <editor-fold desc="Step 4c - Android < 10 location permission">
    private fun checkAndroidBelow10LocationPermissions() {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 4c - Android < 10 location permission")
        AccuratLogger.log(
            AccuratLogger.METHOD_START,
            "$TAG.checkAndroidBelow10LocationPermissions()"
        )
        if (isAndroidBelow10()) {
            runAndroidBelow10()
        } else {
            checkAndroidIs10LocationPermissions()
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.checkAndroidBelow10LocationPermissions()")
    }

    private fun runAndroidBelow10() {
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.runAndroidBelow10()")
        if (hasPrecisePermission()) {
            state.accept(ConsentType.PRECISE_PERMISSION)
        } else {
            state.autoRefuse(ConsentType.PRECISE_PERMISSION)
        }
        if (hasForegroundPermission()) {
            state.accept(ConsentType.IN_APP_LOCATION, ConsentType.ALWAYS_LOCATION)
            save()
            sendConsentsToServer()

            return
        }

        if (state.configuration?.usesLocationPermissionFeature() == false || !state.mayAsk(
                ConsentType.ALWAYS_LOCATION
            )
        ) {
            state.autoRefuse(ConsentType.IN_APP_LOCATION, ConsentType.ALWAYS_LOCATION)
            save()
            sendConsentsToServer()

            return
        }

        showLocationRationale(
            ConsentType.ALWAYS_LOCATION,
            ANDROID_BELOW_10_LOCATION_PERMISSION_REQUESTED
        )
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.runAndroidBelow10()")
    }

    private fun continueLocationPermissionBelowAndroid10() {
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.continueLocationPermissionBelowAndroid10()")
        if (hasPrecisePermission()) {
            state.accept(ConsentType.PRECISE_PERMISSION)
        } else {
            state.autoRefuse(ConsentType.PRECISE_PERMISSION)
        }
        if (hasForegroundPermission()) {
            state.accept(ConsentType.IN_APP_LOCATION, ConsentType.ALWAYS_LOCATION)
        } else {
            state.refuse(ConsentType.IN_APP_LOCATION, ConsentType.ALWAYS_LOCATION)
        }
        save()
        sendConsentsToServer()
        AccuratLogger.log(
            AccuratLogger.METHOD_END,
            "$TAG.continueLocationPermissionBelowAndroid10()"
        )
    }
    // </editor-fold>

    // <editor-fold desc="Step 4d - Android == 10 location permission">
    private fun checkAndroidIs10LocationPermissions() {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 4d - Android == 10 location permission")
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.checkAndroidIs10LocationPermissions()")
        if (isAndroid10()) {
            runAndroid10()
        } else {
            runAndroidAbove10()
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.checkAndroidIs10LocationPermissions()")
    }

    private fun runAndroid10() {
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.runAndroid10()")
        if (hasPrecisePermission()) {
            state.accept(ConsentType.PRECISE_PERMISSION)
        } else {
            state.autoRefuse(ConsentType.PRECISE_PERMISSION)
        }
        if (hasForegroundPermission()) {
            state.accept(ConsentType.IN_APP_LOCATION)
            save()

            if (hasBackgroundPermission()) {
                state.accept(ConsentType.ALWAYS_LOCATION)
                save()
                sendConsentsToServer()

                return
            }

            if (state.configuration?.usesLocationPermissionFeature() == false) {
                state.autoRefuse(ConsentType.ALWAYS_LOCATION)
                save()
                sendConsentsToServer()

                return
            }
        } else if (state.configuration?.usesLocationPermissionFeature() == false) {
            state.autoRefuse(ConsentType.IN_APP_LOCATION, ConsentType.ALWAYS_LOCATION)
            save()
            sendConsentsToServer()

            return
        }

        if (!state.mayAsk(ConsentType.ALWAYS_LOCATION)) {
            state.autoRefuse(ConsentType.IN_APP_LOCATION, ConsentType.ALWAYS_LOCATION)
            save()
            sendConsentsToServer()

            return
        }

        showLocationRationale(ConsentType.ALWAYS_LOCATION, ANDROID_10_LOCATION_PERMISSION_REQUESTED)
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.runAndroid10()")
    }

    private fun continueLocationPermissionAndroid10() {
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.continueLocationPermissionAndroid10()")
        if (hasPrecisePermission()) {
            state.accept(ConsentType.PRECISE_PERMISSION)
        } else {
            state.autoRefuse(ConsentType.PRECISE_PERMISSION)
        }
        if (hasForegroundPermission()) {
            state.accept(ConsentType.IN_APP_LOCATION)
            save()

            if (hasBackgroundPermission()) {
                state.accept(ConsentType.ALWAYS_LOCATION)
                save()
                sendConsentsToServer()

                return
            }
        } else {
            state.refuse(ConsentType.IN_APP_LOCATION)
        }

        state.refuse(ConsentType.ALWAYS_LOCATION)
        save()
        sendConsentsToServer()
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.continueLocationPermissionAndroid10()")
    }
    // </editor-fold>

    // <editor-fold desc="Step 4e - Android > 10 location permission">
    private fun runAndroidAbove10() {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 4e - Android > 10 location permission")
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.runAndroidAbove10()")
        if (hasPrecisePermission()) {
            state.accept(ConsentType.PRECISE_PERMISSION)
        } else {
            state.autoRefuse(ConsentType.PRECISE_PERMISSION)
        }
        if (hasForegroundPermission()) {
            state.accept(ConsentType.IN_APP_LOCATION)
            save()

            if (hasBackgroundPermission()) {
                state.accept(ConsentType.ALWAYS_LOCATION)
                save()
                sendConsentsToServer()

                return
            }

            if (state.configuration?.usesLocationPermissionFeature() != false) {
                prepareAskBackground()

                return
            }

            state.autoRefuse(ConsentType.ALWAYS_LOCATION)
            save()
            sendConsentsToServer()

            return
        }

        if (state.configuration?.usesLocationPermissionFeature() == false || !state.mayAsk(
                ConsentType.IN_APP_LOCATION
            )
        ) {
            state.autoRefuse(ConsentType.IN_APP_LOCATION, ConsentType.ALWAYS_LOCATION)
            save()
            sendConsentsToServer()

            return
        }

        showLocationRationale(
            ConsentType.IN_APP_LOCATION,
            ANDROID_ABOVE_10_FOREGROUND_LOCATION_PERMISSION_REQUESTED
        )
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.runAndroidAbove10()")
    }

    private fun prepareAskBackground() {
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.prepareAskBackground()")
        if (!state.mayAsk(ConsentType.ALWAYS_LOCATION)) {
            state.autoRefuse(ConsentType.ALWAYS_LOCATION)
            save()
            sendConsentsToServer()

            return
        }

        showLocationRationale(
            ConsentType.ANDROID_11_ALWAYS_LOCATION,
            ANDROID_ABOVE_10_BACKGROUND_LOCATION_PERMISSION_REQUESTED
        )
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.prepareAskBackground()")
    }

    private fun continueForegroundLocationPermissionAndroidAbove10() {
        AccuratLogger.log(
            AccuratLogger.METHOD_START,
            "$TAG.continueForegroundLocationPermissionAndroidAbove10()"
        )
        load()
        if (hasPrecisePermission()) {
            state.accept(ConsentType.PRECISE_PERMISSION)
        } else {
            state.autoRefuse(ConsentType.PRECISE_PERMISSION)
        }
        if (hasForegroundPermission()) {
            state.accept(ConsentType.IN_APP_LOCATION)
            save()
            prepareAskBackground()

            return
        }

        state.refuse(ConsentType.IN_APP_LOCATION)
        state.autoRefuse(ConsentType.ALWAYS_LOCATION)
        save()
        sendConsentsToServer()
        AccuratLogger.log(
            AccuratLogger.METHOD_END,
            "$TAG.continueForegroundLocationPermissionAndroidAbove10()"
        )
    }

    private fun continueBackgroundLocationPermissionAndroidAbove10() {
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.continueBackgroundLocationPermissionAndroidAbove10()")
        if (hasPrecisePermission()) {
            state.accept(ConsentType.PRECISE_PERMISSION)
        } else {
            state.autoRefuse(ConsentType.PRECISE_PERMISSION)
        }
        if (hasBackgroundPermission()) {
            state.accept(ConsentType.ALWAYS_LOCATION)
        } else {
            state.refuse(ConsentType.ALWAYS_LOCATION)
        }
        save()
        sendConsentsToServer()
        AccuratLogger.log(
            AccuratLogger.METHOD_END,
            "$TAG.continueBackgroundLocationPermissionAndroidAbove10()"
        )
    }
    // </editor-fold>
    // </editor-fold>

    // <editor-fold desc="Step 5 - Upload consents">
    private fun sendConsentsToServer(isUploadingOldConsents: Boolean = false) {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Step 5 - Upload consents")
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.sendConsentsToServer()")
        state isAt UPLOADING_USER_CONSENTS
        CoroutineScope(Dispatchers.IO).launch {
            val success = uploadUserConsents()
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "Upload consents succeeded? $success")
        }

        if (isUploadingOldConsents) {
            AccuratLogger.log(
                AccuratLogger.SDK_FLOW,
                "Uploaded old user consents, continuing with checking consents for changes"
            )
            checkConsents()

            AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.sendConsentsToServer()")
            return
        }

        state isAt FINISHED
        finish()
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.sendConsentsToServer()")
    }

    /**
     * Push the user consents to the server, without continuing the flow.
     * Useful for when the user consents have been changed programmatically by the implementer of the SDK.
     */
    fun setAndPushConsentsToServer(consentType: ConsentType, isConsentAccepted: Boolean, onComplete: (() -> Unit)? = null) {
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.pushConsentsToServer()")
        CoroutineScope(Dispatchers.Default).launch {
            try {
                AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- Checking Ad ID")
                val adId = loadAdId()
                if (adId == null || adId.isLimitAdTrackingEnabled) {
                    AccuratLogger.log(
                        AccuratLogger.WARNING, "Could not push consents because: ${
                            when {
                                adId == null -> "adId == null"
                                adId.isLimitAdTrackingEnabled -> "adId limit ad tracking is enabled"
                                else -> "unknown reason"
                            }
                        }"
                    )

                    return@launch
                }
                state.adId = adId.id
                state.adIdTrackingAllowed = !adId.isLimitAdTrackingEnabled

                AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- Ad ID is available and usable")
                CoroutineScope(Dispatchers.IO).launch {
                    AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- Downloading user consents")
                    val userConsents: List<UserConsent>? = downloadUserConsents()
                    AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- User consents downloaded")
                    AccuratLogger.log(AccuratLogger.NONE, "-- userConsents = $userConsents")
                    CoroutineScope(Dispatchers.Main).launch {
                        AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- Saving user consents")
                        userConsents?.let {
                            state.setUserConsents(userConsents)
                        }
                        if (isConsentAccepted) {
                            AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- Accepting")
                            state.accept(consentType)
                        } else {
                            AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- Refusing")
                            state.refuse(consentType)
                        }
                        ConsentManagerState.setAppConsent(consentType, isConsentAccepted)
                        AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- userConsents 2 = ${state.userConsents.toList()}")

                        AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- Uploading user consents")
                        withContext(Dispatchers.IO) {
                            val success = uploadUserConsents()
                            CoroutineScope(Dispatchers.Main).launch {
                                AccuratLogger.log(
                                    AccuratLogger.SDK_FLOW,
                                    "Upload consents succeeded? $success"
                                )
                                onComplete?.invoke()
                            }
                        }
                    }
                }
            } catch (e: Exception) {
                AccuratLogger.log(AccuratLogger.ERROR, "Could not push consents: ${e.message}")
            }
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.pushConsentsToServer()")
    }

    private fun finish(success: Boolean? = null) {
        AccuratLogger.log(AccuratLogger.NONE, "finish($success)")
        state isAt NONE
        consentsAskedCallback?.run() ?: startedCallback?.onProcessed(success ?: canStartTracking())
    }

    // <editor-fold desc="API - Upload User Consents">
    private suspend fun uploadUserConsents() = suspendCoroutine<Boolean?> { continuation ->
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.uploadUserConsents()")
        if (!AccuratApi.isNetworkAvailable(context)) {
            AccuratLogger.log(
                AccuratLogger.NETWORK,
                "Network not available, cancelling uploadUserConsents()"
            )
            state.shouldUploadUserConsents = true
            save()
            continuation.resume(null)

            return@suspendCoroutine
        }

        val data = getUploadConsentData()
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "-- data to upload = ${data?.toString()}")
        val request = getUploadConsentsRequest(
            data,
            continuation
        )

        AccuratLogger.logNetworkRequest(
            HttpMethod.POST,
            AccuratEndpoints.POST_CONSENT.url,
            data,
            false
        )
        request.prepare(TAG)
            .enqueue()
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.uploadUserConsents()")
    }

    private fun getUploadConsentsRequest(
        data: JSONObject?,
        continuation: Continuation<Boolean?>,
    ): JsonObjectRequest {
        return object : JsonObjectRequest(
            Method.POST,
            AccuratEndpoints.POST_CONSENT.url,
            data,
            onUploadConsentsResponse(continuation),
            onUploadConsentsError(continuation)
        ) {
            override fun getHeaders(): MutableMap<String, String> = AccuratApi.getHeaders(
                storage,
                "POST",
                bodyContentType,
                AccuratApi.getEncodedRequestBody(data.toString()),
                AccuratEndpoints.POST_CONSENT.path
            )
        }
    }

    private fun getUploadConsentData(): JSONObject? = state.userConsents.withoutDuplicates { userConsent ->
        userConsent.type.value
    }.getUploadData(state.adId)

    private fun onUploadConsentsResponse(continuation: Continuation<Boolean?>): Response.Listener<JSONObject> =
        Response.Listener { response ->
            AccuratLogger.logNetworkResponse(
                HttpMethod.POST,
                Configuration.ENDPOINT_POST_CONSENT,
                response,
                false
            )
            if (response == null || !response.has(ApiKeys.PostConsents.Response.DATA)) {
                AccuratLogger.log(
                    AccuratLogger.ERROR,
                    "No ${ApiKeys.PostConsents.Response.DATA}-field"
                )
                continuation.resume(null)

                return@Listener
            }

            continuation.resume(true)
        }

    private fun onUploadConsentsError(continuation: Continuation<Boolean?>): Response.ErrorListener =
        Response.ErrorListener { error ->
            AccuratLogger.logNetworkError(
                HttpMethod.POST,
                Configuration.ENDPOINT_POST_CONSENT,
                error
            )
            continuation.resume(null)
        }
    // </editor-fold>
    // </editor-fold>

    // <editor-fold desc="Step 6 - Tracking">
    fun canStartTracking(): Boolean {
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.canStartTracking()")
        AccuratLogger.log(AccuratLogger.NONE, "state = $state")
        AccuratLogger.log(AccuratLogger.NONE, "isInitialised() = ${isInitialised()}")
        AccuratLogger.log(AccuratLogger.NONE, "state.startRequested = ${state.startRequested}")
        AccuratLogger.log(AccuratLogger.NONE, "state.userConsents[ConsentType.GDPR] = ${state.userConsents.find(ConsentType.GDPR)}")
        AccuratLogger.log(
            AccuratLogger.NONE,
            "state.check(ConsentType.GDPR) = ${state.check(ConsentType.GDPR)}"
        )
        return isInitialised() && state.startRequested && state.check(ConsentType.GDPR) && hasForegroundPermission()
    }

    fun check(consentType: ConsentType): Boolean = isInitialised() && state.check(consentType)
    // </editor-fold>

    // <editor-fold desc="Info upload">
    private fun sendGeneralInfoToServer() {
        AccuratLogger.log(AccuratLogger.SDK_FLOW, "Upload general info")
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.uploadGeneralInfo()")
        CoroutineScope(Dispatchers.IO).launch {
            AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.uploadGeneralInfo().CoroutineScope")
            val settings = getAccuratSettings()
            val endpoints = settings.endpoints
            if (endpoints == null) {
                AccuratLogger.log(AccuratLogger.ERROR, "Could not parse endpoints")

                return@launch
            }

            if (endpoints.isEmpty()) {
                AccuratLogger.log(AccuratLogger.NONE, "No endpoints to dispatch to")

                return@launch
            }

            val httpClient = HttpClientImpl()
            val storage =
                MultiProcessStorage.getStorage(context, StorageKeys.ACCURAT_MULTI_PROCESS_STORAGE)
            val body = generateGeneralInfo().toString()

            // Send the locations
            AccuratLogger.log(
                AccuratLogger.DISPATCH,
                "Dispatching to " + endpoints.size + " endpoints"
            )
            endpoints.forEach { endpoint ->
                AccuratLogger.log(AccuratLogger.DISPATCH, "Dispatching to " + endpoint.url)
                val endpointUrl = endpoint.url
                AccuratLogger.log(AccuratLogger.NETWORK_REQUEST, "Calling $endpointUrl")
                AccuratLogger.log(AccuratLogger.NETWORK_REQUEST_DATA, body)
                val headers = LocationDispatcher.getEndpointHeaders(storage, endpoint, body)
                httpClient.post(
                    endpointUrl,
                    body,
                    headers,
                    { request, response ->
                        AccuratLogger.logNetworkResponse(
                            HttpMethod.POST,
                            endpointUrl,
                            response,
                            false
                        )
                        AccuratLogger.log(
                            AccuratLogger.DISPATCHER,
                            "Successfully posted 1 location"
                        )
                    },
                    { request, response ->
                        AccuratLogger.logNetworkError(
                            HttpMethod.POST,
                            endpointUrl,
                            response
                        )
                        AccuratLogger.log(
                            AccuratLogger.DISPATCHER,
                            "Failed to post 1 location"
                        )
                    }
                )
            }

            AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.uploadGeneralInfo().CoroutineScope")
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.uploadGeneralInfo()")
    }

    private fun getAccuratSettings(): AccuratSettings {
        return AccuratSettingsManager.getSettings() ?: AccuratSettings.getDefaultSettings().apply {
            addDefaultEndpoint()
        }
    }

    private fun generateGeneralInfo(): JSONObject {
        val generalInfo = AccuratLocationBuilder()
            .setAdvertisingInfo(ConsentManagerState.getAdInfo())
            .setInformationFields(
                InformationFieldsFactory.getInformationFields(
                    Build.MANUFACTURER,
                    Build.MODEL,
                    "",
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) Build.VERSION.BASE_OS else "Android SDK " + Build.VERSION.SDK_INT,
                    "",
                    "",
                    "",
                    "",
                    "",
                    "",
                    null
                )
            )
            .build()

        val jsonObject = JSONObject()
        val generalInfoJson = generalInfo.json.apply {
            try {
                put(AccuratLocationKeys.SDK_VERSION, Configuration.SDK_VERSION)
                put(AccuratLocationKeys.APP_VERSION, AccuratConfigurationManager.getAppVersion())
                put(
                    AccuratLocationKeys.APP_VERSION_CODE,
                    AccuratConfigurationManager.getAppVersionCode().toString()
                )
            } catch (e: JSONException) {
                e.printStackTrace()
            }
        }
        val jsonArray = JSONArray().apply {
            put(generalInfoJson)
        }

        try {
            jsonObject.put(LocationDispatcher.LOCATIONS_KEY, jsonArray)
        } catch (e: JSONException) {
            AccuratLogger.log(AccuratLogger.JSON_ERROR, "$TAG.generateGeneralInfo(): ${e.message}")
        }

        return jsonObject
    }
    // </editor-fold>

    // <editor-fold desc="Helpers">
    private fun save() {
        AccuratLogger.log(AccuratLogger.STORAGE, "Saving ConsentManagerState:")
        AccuratLogger.log(AccuratLogger.STORAGE_DATA, state.toString())
        state.save()
    }

    private fun load() {
        AccuratLogger.log(AccuratLogger.STORAGE, "Loading ConsentManagerState:")
        state = ConsentManagerState.load()
        AccuratLogger.log(AccuratLogger.STORAGE_DATA, state.toString())
    }

    private fun isInitialised(): Boolean = ::context.isInitialized

    private fun enqueueRequest(request: JsonObjectRequest) {
        if (requestQueue == null) {
            requestQueue = Volley.newRequestQueue(context)
        }

        requestQueue?.apply {
            add(request)
        }
    }

    // <editor-fold desc="Decisions">
    private fun shouldAskConsents(): Boolean {
        load()
        val shouldAsk = state.forceAskConsents || state.isValidAskConsentsTiming()
        val reason = when {
            state.forceAskConsents -> "forceAskConsents is true"
            state.isValidAskConsentsTiming() -> "ask consent timing is valid"
            else -> "forceAskConsents is false and ask consent timing is invalid"
        }
        AccuratLogger.log(
            AccuratLogger.SDK_FLOW,
            "shouldAskConsents? ${if (shouldAsk) "yes" else "no"} because $reason"
        )

        return shouldAsk
    }
    // </editor-fold>

    // <editor-fold desc="Permissions">
    private fun hasPermission(permission: String): Boolean =
        ActivityCompat.checkSelfPermission(context, permission) == PackageManager.PERMISSION_GRANTED

    private fun hasForegroundPermission(): Boolean {
        val hasPermission =
            hasPermission(Manifest.permission.ACCESS_COARSE_LOCATION) || hasPermission(Manifest.permission.ACCESS_FINE_LOCATION)
        AccuratLogger.log(AccuratLogger.NONE, "hasForegroundPermission? $hasPermission")

        return hasPermission
    }

    private fun hasPrecisePermission(): Boolean {
        val hasPermission = hasPermission(Manifest.permission.ACCESS_FINE_LOCATION)
        AccuratLogger.log(AccuratLogger.NONE, "hasPrecisePermission? $hasPermission")

        return hasPermission
    }

    @SuppressLint("InlinedApi")
    private fun hasBackgroundPermission(): Boolean {
        val hasPermission =
            isAndroidBelow10() || hasPermission(Manifest.permission.ACCESS_BACKGROUND_LOCATION)
        AccuratLogger.log(AccuratLogger.NONE, "hasBackgroundPermission? $hasPermission")

        return hasPermission
    }

    private fun showLocationRationale(consentType: ConsentType, requestCode: Int) {
        AccuratLogger.log(AccuratLogger.METHOD_START, "$TAG.showLocationRationale($consentType)")
        val viewHolder =
            ConsentDialogViewHolder(View.inflate(context, R.layout.dialog_consent, null))
        viewHolder.setConsentModel(state.consentModels.find(consentType))

        val consentDialog = AlertDialog.Builder(context)
            .setView(viewHolder.rootView)
            .setCancelable(false)
            .create()
        consentDialog.setCanceledOnTouchOutside(false)

        viewHolder.setPositiveButtonHandler {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "LocationRationaleDialog.onPositive()")
            consentDialog.dismiss()

            askLocationPermission(consentType, requestCode)
        }
        viewHolder.setNegativeButtonHandler {
            AccuratLogger.log(AccuratLogger.SDK_FLOW, "LocationRationaleDialog.onNegative()")
            consentDialog.dismiss()

            when (requestCode) {
                ANDROID_BELOW_10_LOCATION_PERMISSION_REQUESTED -> continueLocationPermissionBelowAndroid10()
                ANDROID_10_LOCATION_PERMISSION_REQUESTED -> continueLocationPermissionAndroid10()
                ANDROID_ABOVE_10_FOREGROUND_LOCATION_PERMISSION_REQUESTED -> continueForegroundLocationPermissionAndroidAbove10()
                ANDROID_ABOVE_10_BACKGROUND_LOCATION_PERMISSION_REQUESTED -> continueBackgroundLocationPermissionAndroidAbove10()
            }
        }

        try {
            if (!(context as Activity).isFinishing) {
                consentDialog.show()
                viewHolder.enableDialogLinks()
            } else {
                AccuratLogger.log(
                    AccuratLogger.WARNING,
                    "Could not show LocationRationaleDialog: Activity is finishing"
                )
            }
        } catch (e: Exception) {
            AccuratLogger.log(
                AccuratLogger.ERROR,
                "Could not show LocationRationaleDialog: ${e.message}"
            )
        }
        AccuratLogger.log(AccuratLogger.METHOD_END, "$TAG.showLocationRationale($consentType)")
    }

    private fun askLocationPermission(consentType: ConsentType, requestCode: Int) {
        AccuratLogger.log(
            AccuratLogger.SDK_FLOW,
            "askLocationPermission($consentType, ${consentType.systemPermissions}, ${
                requestCodeString(requestCode)
            })"
        )
        when (requestCode) {
            ANDROID_BELOW_10_LOCATION_PERMISSION_REQUESTED -> state isAt ConsentFlowState.ANDROID_BELOW_10_LOCATION_PERMISSION_REQUESTED
            ANDROID_10_LOCATION_PERMISSION_REQUESTED -> state isAt ConsentFlowState.ANDROID_10_LOCATION_PERMISSION_REQUESTED
            ANDROID_ABOVE_10_FOREGROUND_LOCATION_PERMISSION_REQUESTED -> state isAt ConsentFlowState.ANDROID_ABOVE_10_FOREGROUND_LOCATION_PERMISSION_REQUESTED
            ANDROID_ABOVE_10_BACKGROUND_LOCATION_PERMISSION_REQUESTED -> state isAt ConsentFlowState.ANDROID_ABOVE_10_BACKGROUND_LOCATION_PERMISSION_REQUESTED
        }
        ConsentManagerState.updateConsentRequestTimestamp()
        load()
        ActivityCompat.requestPermissions(
            activity,
            consentType.systemPermissions,
            requestCode
        )
    }

    fun onRequestPermissionResult(
        requestCode: Int,
        permissions: Array<String>,
        grantResults: IntArray,
        context: Context,
    ): Boolean {
        AccuratLogger.log(
            AccuratLogger.SDK_FLOW,
            "onRequestPermissionResult(${requestCodeString(requestCode)}, $permissions, $grantResults)"
        )
        when (requestCode) {
            ANDROID_BELOW_10_LOCATION_PERMISSION_REQUESTED -> continueLocationPermissionBelowAndroid10()
            ANDROID_10_LOCATION_PERMISSION_REQUESTED -> continueLocationPermissionAndroid10()
            ANDROID_ABOVE_10_FOREGROUND_LOCATION_PERMISSION_REQUESTED -> continueForegroundLocationPermissionAndroidAbove10()
            ANDROID_ABOVE_10_BACKGROUND_LOCATION_PERMISSION_REQUESTED -> continueBackgroundLocationPermissionAndroidAbove10()
            else -> return false
        }

        return true
    }

    private fun requestCodeString(requestCode: Int) = when (requestCode) {
        ANDROID_BELOW_10_LOCATION_PERMISSION_REQUESTED -> "ANDROID_BELOW_10_LOCATION_PERMISSION_REQUESTED"
        ANDROID_10_LOCATION_PERMISSION_REQUESTED -> "ANDROID_10_LOCATION_PERMISSION_REQUESTED"
        ANDROID_ABOVE_10_FOREGROUND_LOCATION_PERMISSION_REQUESTED -> "ANDROID_ABOVE_10_FOREGROUND_LOCATION_PERMISSION_REQUESTED"
        ANDROID_ABOVE_10_BACKGROUND_LOCATION_PERMISSION_REQUESTED -> "ANDROID_ABOVE_10_BACKGROUND_LOCATION_PERMISSION_REQUESTED"
        else -> requestCode.toString()
    }
    // </editor-fold>

    // <editor-fold desc="Android versions">
    private fun androidVersion(): Int = Build.VERSION.SDK_INT

    private fun android10(): Int = Build.VERSION_CODES.Q

    private fun isAndroidBelow10(): Boolean = androidVersion() < android10()

    private fun isAndroid10(): Boolean = androidVersion() == android10()

    private fun isAndroidAbove10(): Boolean = androidVersion() > android10()
    // </editor-fold>
    // </editor-fold>

    // <editor-fold desc="Extension functions">
    private fun JsonObjectRequest.prepare(tag: String): JsonObjectRequest {
        setTag(tag)
        retryPolicy = AccuratApi.defaultRetryPolicy
        setShouldCache(false)

        return this
    }

    private fun JsonObjectRequest.enqueue() {
        enqueueRequest(this)
    }

    private fun <T, S> RealmList<T>.withoutDuplicates(selector: (T) -> S): List<T> {
        val set = mutableMapOf<S, T>()
        toList().forEach { item ->
            set[selector(item)] = item
        }

        return set.values.toList()
    }
    // </editor-fold>
}